import React, { useCallback, useContext, useEffect, useState } from 'react';
import { FormSection } from '@atlaskit/form';

import { BitbucketPermission, bitbucketPermissions } from '../permissions';
import PermissionOptions from './options/PermissionOptions';
import { PermissionsContext, ScopeDescriptions } from '../PermissionsContext';
import BitbucketPermissionDescriptions from './BitbucketPermissionDescriptions';

export const initialSelectedPermissions: Record<string, boolean> = {
    [BitbucketPermission.PUBLIC_REPOS]: true,
    [BitbucketPermission.PROJECT_ADMIN]: false,
    [BitbucketPermission.REPO_ADMIN]: false,
    [BitbucketPermission.REPO_READ]: false,
    [BitbucketPermission.REPO_WRITE]: false,
    [BitbucketPermission.ACCOUNT_WRITE]: false,
    [BitbucketPermission.PUBLIC_REPOS]: true,
    [BitbucketPermission.SYSTEM_ADMIN]: false,
    [BitbucketPermission.ADMIN_WRITE]: false,
};

interface CheckedPermissions {
    [key: string]: boolean;
}

interface DisabledPermissions {
    [key: string]: boolean;
}

function populateScopesUIFromExistingConfig(
    scopes: string | undefined,
    changeToChecked: (scope: string) => void
) {
    if (scopes !== undefined) {
        scopes.split(' ').forEach((scope) => {
            changeToChecked(scope);
        });
    }
}

export function getBitbucketPermissionRank(permission: string): number | undefined {
    switch (permission) {
        case BitbucketPermission.PUBLIC_REPOS:
            return 0;
        case BitbucketPermission.ACCOUNT_WRITE:
            return 1;
        case BitbucketPermission.REPO_READ:
            return 2;
        case BitbucketPermission.REPO_WRITE:
            return 3;
        case BitbucketPermission.REPO_ADMIN:
            return 4;
        case BitbucketPermission.PROJECT_ADMIN:
            return 5;
        case BitbucketPermission.ADMIN_WRITE:
            return 6;
        case BitbucketPermission.SYSTEM_ADMIN:
            return 7;
        default:
            return undefined;
    }
}

function sortPermissions(permissionA: string, permissionB: string): number {
    const rankA = getBitbucketPermissionRank(permissionA) ?? 0;
    const rankB = getBitbucketPermissionRank(permissionB) ?? 0;
    return rankA - rankB;
}

function getPermissionDescriptions(
    checkedPermissions: CheckedPermissions,
    descriptions: ScopeDescriptions | undefined
): string[] {
    if (descriptions === undefined) {
        return [];
    }

    return Object.keys(checkedPermissions)
        .filter((permission) => checkedPermissions[permission])
        .sort(sortPermissions)
        .reduce<string[]>((collectedPermissionDescriptions, permission) => {
            const permissionDescriptions = descriptions[permission]?.descriptions;
            if (permissionDescriptions) {
                return [...collectedPermissionDescriptions, ...permissionDescriptions];
            }
            return collectedPermissionDescriptions;
        }, []);
}

interface PermissionsBitbucketProps {
    scopes?: string;
}

export default function PermissionsBitbucket({ scopes }: PermissionsBitbucketProps) {
    const { descriptions } = useContext(PermissionsContext);
    const [checkedPermissions, setCheckedPermissions] = useState<CheckedPermissions>({
        ...initialSelectedPermissions,
    });
    const [disabledPermissions, setDisabledPermissions] = useState<DisabledPermissions>({
        ...initialSelectedPermissions,
    });

    const [permissionDescriptions, setPermissionDescriptions] = useState<string[]>([]);

    useEffect(() => {
        if (descriptions !== undefined) {
            setPermissionDescriptions([
                ...getPermissionDescriptions(checkedPermissions, descriptions.scopes),
            ]);
        }
    }, [checkedPermissions, descriptions]);

    const updateAssociatedPermissions = useCallback(
        (permission: string, isChecked: boolean) => {
            const perms = bitbucketPermissions.get(permission);

            if (perms !== undefined) {
                perms.forEach((associatedPermission: string) => {
                    checkedPermissions[associatedPermission] = isChecked;
                    disabledPermissions[associatedPermission] = isChecked;
                });
                setDisabledPermissions({ ...disabledPermissions });
                setCheckedPermissions({ ...checkedPermissions });
            }
        },
        [checkedPermissions, disabledPermissions]
    );

    const changeToChecked = useCallback(
        (permission: string) => {
            checkedPermissions[permission] = true;
            updateAssociatedPermissions(permission, true);
        },
        [checkedPermissions, updateAssociatedPermissions]
    );

    useEffect(() => {
        if (scopes !== undefined) {
            populateScopesUIFromExistingConfig(scopes, changeToChecked);
        }
    }, [scopes, changeToChecked]);

    function changeToUnchecked(permission: string) {
        checkedPermissions[permission] = false;
        setCheckedPermissions({ ...checkedPermissions });
        updateAssociatedPermissions(permission, false);
    }

    const handleChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        if (descriptions !== undefined) {
            const clickedPermission = event.target.value;
            if (!checkedPermissions[clickedPermission]) {
                changeToChecked(clickedPermission);
            } else if (checkedPermissions[clickedPermission]) {
                changeToUnchecked(clickedPermission);
            }
            setPermissionDescriptions([
                ...getPermissionDescriptions(checkedPermissions, descriptions.scopes),
            ]);
        }
    };

    return (
        <FormSection>
            <PermissionOptions
                handleChange={handleChange}
                checkedPermissions={checkedPermissions}
                disabledPermissions={disabledPermissions}
            />
            {permissionDescriptions && (
                <BitbucketPermissionDescriptions descriptions={permissionDescriptions} />
            )}
        </FormSection>
    );
}
