import { act, fireEvent } from '@testing-library/react';
import { BambooPermission, BasicPermission, BitbucketPermission } from '../permissions';
import { PermissionsContext } from '../PermissionsContext';

export function expectI18nKeysToRender(i18nKeys, getByText, ignore = []) {
    i18nKeys.forEach((i18nKey) => {
        if (ignore.includes(i18nKey) || typeof i18nKey === 'function') {
            return;
        }

        expect(getByText(i18nKey)).toBeTruthy();
    });
}

export const updateFormField = (field, newValue) =>
    fireEvent.change(field, {
        target: { value: newValue },
    });

export const untilPromisesResolved = () => new Promise((resolve) => setImmediate(resolve));

export async function waitForNetworkCallsToFinish() {
    await act(async () => {
        await untilPromisesResolved();
    });
}

export const readPermissionTitle = 'View projects and issues';
export const readPermissionDesc =
    'View projects and issues your account can view, including any related items, such as dashboards, filters, attachments, or comments. Also view your user profile.';

export const writePermissionTitle = 'Create, update, and delete projects and issues';
export const writePermissionDesc =
    'Create, update, and delete projects and issues your account can change, including any related items, such as dashboards, filters, attachments, or comments. Also change your user profile.';

export const withBasicPermissionDescriptions = {
    descriptions: {
        consent: {
            READ: { title: readPermissionTitle, description: readPermissionDesc },
            WRITE: { title: writePermissionTitle, description: writePermissionDesc },
        },
    },
};

export const basicPermissions = {
    [BasicPermission.READ]: {
        title: 'View projects and issues',
        description:
            'View projects and issues your account can view, including any related items, such as dashboards, filters, attachments, or comments. Also view your user profile.',
    },
    [BasicPermission.WRITE]: {
        title: 'Create, update, and delete projects and issues',
        description:
            'Create, update, and delete projects and issues your account can change, including any related items, such as dashboards, filters, attachments, or comments. Also change your user profile.',
    },
    [BasicPermission.ADMIN]: {
        title: 'Administer Jira',
        description:
            'Perform most administrative functions on the entire Jira instance, excluding functions such as backups, imports, and security settings which are limited to system administrators.',
    },
    [BasicPermission.SYSTEM_ADMIN]: {
        title: 'Administer Jira system',
        description:
            'Perform all administrative functions on the entire Jira instance, including functions such as backups, imports, and security settings.',
    },
};

export const bambooPermissions = {
    [BambooPermission.READ]: {
        title: 'Read permissions',
        description: 'The token will be allowed to read data from Bamboo.',
    },
    [BambooPermission.TRIGGER]: {
        title: 'Triggering permissions',
        description: 'Allows the token to trigger builds and deployments.',
    },
    [BambooPermission.USER]: {
        title: 'Same as user',
        description: 'The token will have the same permissions as you do.',
    },
};

export const bitbucketPermissions = {
    [BitbucketPermission.PUBLIC_REPOS]: {
        descriptions: ['public repos description'],
    },
    [BitbucketPermission.ACCOUNT_WRITE]: {
        descriptions: ['account write description'],
    },
    [BitbucketPermission.REPO_READ]: {
        descriptions: ['repo read primary description', 'repo read secondary description'],
    },
    [BitbucketPermission.REPO_WRITE]: { descriptions: ['repo write description'] },
    [BitbucketPermission.REPO_ADMIN]: {
        descriptions: ['repo admin primary description', 'repo admin secondary description'],
    },
    [BitbucketPermission.PROJECT_ADMIN]: {
        descriptions: ['project admin primary description', 'project admin secondary description'],
    },
    [BitbucketPermission.ADMIN_WRITE]: { descriptions: ['admin write description'] },
    [BitbucketPermission.SYSTEM_ADMIN]: { descriptions: ['system admin description'] },
};

export const bitbucketPermissionConsent = {
    [BitbucketPermission.PUBLIC_REPOS]: {
        title: 'Public repos',
        description: 'public repos description',
    },
    [BitbucketPermission.ACCOUNT_WRITE]: {
        title: 'Account write',
        description: 'account write description',
    },
    [BitbucketPermission.REPO_READ]: {
        title: 'Repo read',
        description: 'repo read primary description',
    },
    [BitbucketPermission.REPO_WRITE]: {
        title: 'Repo write',
        description: 'repo write description',
    },
    [BitbucketPermission.REPO_ADMIN]: {
        title: 'Repo admin',
        description: 'repo admin primary description',
    },
    [BitbucketPermission.PROJECT_ADMIN]: {
        title: 'Project admin',
        description: 'project admin primary description',
    },
    [BitbucketPermission.ADMIN_WRITE]: {
        title: 'Admin write',
        description: 'admin write description',
    },
    [BitbucketPermission.SYSTEM_ADMIN]: {
        title: 'System admin',
        description: 'system admin description',
    },
};

export const withBitbucketPermissionDescriptions = {
    descriptions: {
        scopes: bitbucketPermissions,
        consent: bitbucketPermissionConsent,
    },
};

export function withContext(contextValue) {
    return (component) => (
        <PermissionsContext.Provider value={contextValue}>{component}</PermissionsContext.Provider>
    );
}
