import { ErrorMessage, Field } from '@atlaskit/form';
import { Box } from '@atlaskit/primitives';
import * as wrmReactI18n from '@atlassian/wrm-react-i18n';
import Select from '@atlaskit/select';
import React, { Fragment } from 'react';

interface OutgoingScopeFieldProps {
    options: string[];
    defaultValue: string[];
    isReadOnly: boolean;
}

const mapToSelectOption = (options: string[]) =>
    (options || []).map((value) => ({ label: value, value }));

export function OutgoingScopeField({ options, defaultValue, isReadOnly }: OutgoingScopeFieldProps) {
    const effectiveDefaultValue = isReadOnly ? options : defaultValue;

    const selectOptions = mapToSelectOption(options);
    const selectDefaultValues = mapToSelectOption(effectiveDefaultValue);

    const validateHasAtLeastOneSelection = (
        selected: Array<{ label: string; value: string }> | undefined
    ) => {
        if (!selected || selected.length === 0) {
            return wrmReactI18n.I18n.getText('applinks-atlassian-oauth2-ui.scope.error');
        }
        return undefined;
    };

    return (
        <Field
            aria-required={true}
            name="outgoingScopes"
            label={wrmReactI18n.I18n.getText('applinks-atlassian-oauth2-ui.scope')}
            isRequired={true}
            validate={validateHasAtLeastOneSelection}
            defaultValue={selectDefaultValues}
        >
            {({ fieldProps, error }) => (
                <Fragment>
                    {/* @ts-expect-error TS2322: Type - onChange params type difference */}
                    <Select
                        {...fieldProps}
                        isInvalid={!!error}
                        inputId="outgoingScopeSelect"
                        options={selectOptions}
                        isMulti
                        isSearchable
                        id={'outgoingScopes-select-id'}
                    />
                    {error && (
                        <ErrorMessage>
                            <Box aria-live="polite" testId="scope-outgoing-error">
                                {wrmReactI18n.I18n.getText(error)}
                            </Box>
                        </ErrorMessage>
                    )}
                </Fragment>
            )}
        </Field>
    );
}
