import React, { Fragment, useEffect } from 'react';
import { AtlassianOauth2UiDialog } from '../AtlassianOauth2UiDialog';
import { I18n } from '@atlassian/wrm-react-i18n';
import SectionMessage from '@atlaskit/section-message';
import Spinner from '@atlaskit/spinner';
import { ClientIdField } from './ClientIdField';
import { ClientSecretField } from './ClientSecretField';
import { LoadingDetailsErrorMessage } from '../LoadingDetailsErrorMessage';
import { performGetIncomingLinkDetails } from './incomingLinkDetailsDialogUtils';
import { dispatchApplinkDialogClosedEvent } from '../atlassianOauth2UiEvents';

export const incomingLinkDetailsDialogInitialState = {
    isOpen: false,
    incomingApplicationLinkDetails: null,
};

const ILD_MODE_CREATED = 'ILD_MODE_CREATED';
const ILD_MODE_VIEW = 'ILD_MODE_VIEW';

export interface IncomingApplicationLinkDetails {
    id: string;
    // ...other fields as needed...
}

export interface IncomingLinkDetailsDialogSubstate {
    isOpen: boolean;
    incomingApplicationLinkDetails: IncomingApplicationLinkDetails | null;
    needToLoadDetails?: boolean;
    hasErrorLoadingDetails?: boolean;
    applinkId?: string;
    mode?: string;
}

export interface IncomingLinkDetailsDialogProps {
    substate: IncomingLinkDetailsDialogSubstate;
    dispatch: React.Dispatch<any>;
    onClose: () => void;
}

export function incomingLinkDetailsDialogReducer(
    substate: IncomingLinkDetailsDialogSubstate,
    action: any
): IncomingLinkDetailsDialogSubstate {
    switch (action.type) {
        case 'ild:openWithDetails': {
            return {
                ...substate,
                isOpen: true,
                incomingApplicationLinkDetails: action.incomingApplicationLinkDetails,
                mode: ILD_MODE_CREATED,
            };
        }
        case 'ild:openFromEvent': {
            const { detail } = action;
            const { applinkId } = detail;
            return {
                ...substate,
                isOpen: true,
                needToLoadDetails: true,
                mode: ILD_MODE_VIEW,
                applinkId,
            };
        }
        case 'ild:loadingSuccessful': {
            return {
                ...substate,
                needToLoadDetails: false,
                incomingApplicationLinkDetails: action.incomingApplicationLinkDetails,
            };
        }
        case 'ild:loadingFailed': {
            return {
                ...substate,
                needToLoadDetails: false,
                hasErrorLoadingDetails: true,
            };
        }
        case 'ild:close': {
            dispatchApplinkDialogClosedEvent();
            return {
                ...substate,
                isOpen: false,
            };
        }
    }
    return substate;
}

export function ConfirmationMessage() {
    return (
        <SectionMessage
            appearance="information"
            title={I18n.getText(
                'applinks-atlassian-oauth2-ui.incoming-link-creation-confirmation.title'
            )}
        >
            <p>
                {I18n.getText(
                    'applinks-atlassian-oauth2-ui.incoming-link-creation-confirmation.description'
                )}
            </p>
        </SectionMessage>
    );
}

export function IncomingLinkDetailsDialog(props: IncomingLinkDetailsDialogProps) {
    const { substate, dispatch } = props;

    const {
        applinkId,
        needToLoadDetails,
        hasErrorLoadingDetails,
        incomingApplicationLinkDetails,
        mode,
    } = substate;

    function handleSubmit() {
        dispatch({
            type: 'ild:close',
        });

        dispatch({
            type: 'col:open',
            detail: {
                applinkId: incomingApplicationLinkDetails?.id,
            },
        });
    }

    function closeModal() {
        props.onClose();
    }

    useEffect(() => {
        if (needToLoadDetails) {
            performGetIncomingLinkDetails(applinkId, dispatch);
        }
    }, [applinkId, needToLoadDetails, dispatch]);

    return (
        <AtlassianOauth2UiDialog
            id="AtlassianOauth2Ui_IncomingLinkDetailsDialog"
            onClose={closeModal}
            onSubmit={handleSubmit}
            titleText={I18n.getText('applinks-atlassian-oauth2-ui.incoming-link-details')}
            primaryButtonIsHidden={mode === ILD_MODE_VIEW}
            primaryButtonText={I18n.getText(
                'applinks-atlassian-oauth2-ui.dialog.configure-outgoing'
            )}
            closeButtonText={I18n.getText('applinks-atlassian-oauth2-ui.dialog.close')}
        >
            {mode === ILD_MODE_CREATED ? <ConfirmationMessage /> : null}
            {needToLoadDetails ? (
                <p>
                    <Spinner />
                    &nbsp;
                    {I18n.getText('applinks-atlassian-oauth2-ui.loading-link-data')}
                </p>
            ) : hasErrorLoadingDetails ? (
                <LoadingDetailsErrorMessage />
            ) : incomingApplicationLinkDetails ? (
                <Fragment>
                    <ClientIdField
                        incomingApplicationLinkDetails={incomingApplicationLinkDetails}
                    />
                    <ClientSecretField
                        incomingApplicationLinkDetails={incomingApplicationLinkDetails}
                    />
                </Fragment>
            ) : null}
        </AtlassianOauth2UiDialog>
    );
}
