import React from 'react';
import { AtlassianOauth2UiDialog } from '../AtlassianOauth2UiDialog';
import { I18n } from '@atlassian/wrm-react-i18n';
import { postResetCredentials } from './ResetCredentialsRest';
import { StyledParagraph } from '../common/StyledParagraph';
import { useFlags } from '@atlaskit/flag';
import SuccessIcon from '@atlaskit/icon/glyph/check-circle';
import ErrorIcon from '@atlaskit/icon/glyph/error';
import { token } from '@atlaskit/tokens';

export interface ResetCredentialsDialogSubstate {
    isOpen: boolean;
    needToLoadDetails?: boolean;
    isResetting?: boolean;
    hasResetError?: boolean;
    hasResetSuccess?: boolean;
    clientId?: string | null;
    applinkId?: string | null;
}

export interface ResetCredentialsDialogProps {
    substate: ResetCredentialsDialogSubstate;
    dispatch: React.Dispatch<any>;
    onClose: () => void;
}

export function ResetCredentialsDialog(props: ResetCredentialsDialogProps) {
    const { substate, dispatch } = props;
    const { showFlag } = useFlags();

    const { isResetting, clientId, applinkId } = substate;

    function showSuccessFlag() {
        showFlag({
            id: 'reset-credentials-success-flag',
            title: I18n.getText('applinks-atlassian-oauth2-ui.reset.success'),
            actions: [
                {
                    content: I18n.getText('applinks-atlassian-oauth2-ui.view-credentials'),
                    testId: 'reset-credentials-view-credentials',
                    onClick: () => {
                        dispatch({
                            type: 'ild:openFromEvent',
                            detail: { applinkId: applinkId ?? undefined },
                        });
                    },
                },
            ],
            icon: (
                <SuccessIcon
                    primaryColor={token('color.icon.success')}
                    label={I18n.getText('applinks-atlassian-oauth2-ui.reset.success')}
                />
            ),
        });
    }

    function showErrorFlag() {
        showFlag({
            id: 'reset-credentials-error-flag',
            title: I18n.getText('applinks-atlassian-oauth2-ui.reset.error'),
            icon: (
                <ErrorIcon
                    primaryColor={token('color.icon.danger')}
                    label={I18n.getText('applinks-atlassian-oauth2-ui.reset.error')}
                />
            ),
            actions: [
                {
                    content: I18n.getText('applinks-atlassian-oauth2-ui.try-again'),
                    testId: 'reset-credentials-try-again',
                    onClick: () => {
                        handleSubmit();
                    },
                },
            ],
        });
    }

    async function handleSubmit() {
        dispatch({
            type: 'rcd:resetStart',
        });

        try {
            await postResetCredentials(clientId);
            dispatch({
                type: 'rcd:resetSuccess',
            });
            showSuccessFlag();
        } catch (error) {
            dispatch({
                type: 'rcd:resetError',
                error,
            });
            showErrorFlag();
        } finally {
            props.onClose();
        }
    }

    function closeModal() {
        props.onClose();
    }

    return (
        <AtlassianOauth2UiDialog
            id="AtlassianOauth2Ui_ResetCredentialsDialog"
            onClose={closeModal}
            onSubmit={handleSubmit}
            titleText={I18n.getText('applinks-atlassian-oauth2-ui.reset.dialog.title')}
            primaryButtonText={I18n.getText('applinks-atlassian-oauth2-ui.reset.button')}
            primaryButtonIsLoading={isResetting}
            primaryButtonAppearance="warning"
            closeButtonText={I18n.getText('applinks-atlassian-oauth2-ui.dialog.close')}
        >
            <StyledParagraph>
                {I18n.getText('applinks-atlassian-oauth2-ui.reset.description.1', clientId ?? '')}
            </StyledParagraph>
            <StyledParagraph>
                {I18n.getText('applinks-atlassian-oauth2-ui.reset.description.2')}
            </StyledParagraph>
        </AtlassianOauth2UiDialog>
    );
}
