import React from 'react';
import { AtlassianOauth2UiDialog } from '../AtlassianOauth2UiDialog';
import { I18n } from '@atlassian/wrm-react-i18n';
import { postRotateCredentials } from './RotateCredentialsRest';
import { StyledParagraph } from '../common/StyledParagraph';
import { useFlags } from '@atlaskit/flag';
import SuccessIcon from '@atlaskit/icon/glyph/check-circle';
import ErrorIcon from '@atlaskit/icon/glyph/error';
import { token } from '@atlaskit/tokens';

export interface RotateCredentialsDialogSubstate {
    isOpen: boolean;
    needToLoadDetails?: boolean;
    hasErrorLoadingDetails?: boolean;
    isRotating?: boolean;
    hasRotationError?: boolean;
    hasRotationSuccess?: boolean;
    expiryDate?: string | null;
    clientId?: string | null;
    applinkId?: string | null;
}

export interface RotateCredentialsDialogProps {
    substate: RotateCredentialsDialogSubstate;
    dispatch: React.Dispatch<any>;
    onClose: () => void;
}

export function RotateCredentialsDialog(props: RotateCredentialsDialogProps) {
    const { substate, dispatch } = props;
    const { showFlag } = useFlags();

    const { isRotating, expiryDate, clientId, applinkId } = substate;

    function showSuccessFlag() {
        showFlag({
            id: 'rotate-credentials-success-flag',
            title: I18n.getText('applinks-atlassian-oauth2-ui.rotate.success'),
            actions: [
                {
                    content: I18n.getText('applinks-atlassian-oauth2-ui.view-credentials'),
                    testId: 'rotate-credentials-view-credentials',
                    onClick: () => {
                        props.onClose();
                        dispatch({
                            type: 'ild:openFromEvent',
                            detail: { applinkId: applinkId ?? undefined },
                        });
                    },
                },
            ],
            icon: (
                <SuccessIcon
                    primaryColor={token('color.icon.success')}
                    label={I18n.getText('entitylinks.page.flag.rotate.success')}
                />
            ),
        });
    }

    function showErrorFlag() {
        showFlag({
            id: 'rotate-credentials-error-flag',
            title: I18n.getText('applinks-atlassian-oauth2-ui.rotate.error'),
            icon: (
                <ErrorIcon
                    primaryColor={token('color.icon.danger')}
                    label={I18n.getText('applinks-atlassian-oauth2-ui.rotate.error')}
                />
            ),
            actions: [
                {
                    content: I18n.getText('applinks-atlassian-oauth2-ui.try-again'),
                    testId: 'rotate-credentials-try-again',
                    onClick: () => {
                        handleSubmit();
                    },
                },
            ],
        });
    }

    async function handleSubmit() {
        dispatch({
            type: 'rcd:rotateStart',
        });

        try {
            await postRotateCredentials(clientId);
            dispatch({
                type: 'rcd:rotateSuccess',
            });
            showSuccessFlag();
        } catch (error) {
            dispatch({
                type: 'rcd:rotateError',
                error,
            });
            showErrorFlag();
        } finally {
            props.onClose();
        }
    }

    return (
        <AtlassianOauth2UiDialog
            id="AtlassianOauth2Ui_RotateCredentialsDialog"
            onClose={props.onClose}
            onSubmit={handleSubmit}
            titleText={I18n.getText('applinks-atlassian-oauth2-ui.rotate.dialog.title')}
            primaryButtonText={I18n.getText('applinks-atlassian-oauth2-ui.rotate.button')}
            primaryButtonIsLoading={isRotating}
            closeButtonText={I18n.getText('applinks-atlassian-oauth2-ui.dialog.close')}
            modalAppearance={null}
        >
            <StyledParagraph>
                {I18n.getText(
                    'applinks-atlassian-oauth2-ui.rotate.description.1',
                    clientId ?? '',
                    expiryDate ?? '-'
                )}
            </StyledParagraph>
            <StyledParagraph>
                {I18n.getText('applinks-atlassian-oauth2-ui.rotate.description.2')}
            </StyledParagraph>
        </AtlassianOauth2UiDialog>
    );
}
