import React, { Component } from 'react';

import Button from '@atlaskit/button/new';
import DropdownMenu, { DropdownItem, DropdownItemGroup } from '@atlaskit/dropdown-menu';
import MoreIcon from '@atlaskit/icon/glyph/more';
import { ModalTransition } from '@atlaskit/modal-dialog';
import { I18n } from '@atlassian/wrm-react-i18n';
import DeleteEntityLinkDialogForm from './delete-entity-link-dialog';
import { makeEntityLinkPrimary } from './EntitylinksRest';

type EntityLink = {
    key: string;
    applicationId: string;
    name: string;
    typeId: string;
    isPrimary?: boolean;
};

interface EntityLinksMoreActionsProps {
    currentApp: string;
    projectKey: string;
    type: string;
    entityLink: EntityLink;
    outgoingApplicationName: string;
    onDelete: (entityLink: EntityLink) => void;
    onMakePrimary: (entityLink: EntityLink) => void;
}

interface EntityLinksMoreActionsState {
    isDeleteLinkModalOpen: boolean;
}

export default class EntityLinksMoreActions extends Component<
    EntityLinksMoreActionsProps,
    EntityLinksMoreActionsState
> {
    constructor(props: EntityLinksMoreActionsProps) {
        super(props);
        this.state = {
            isDeleteLinkModalOpen: false,
        };
    }

    onClickMakePrimaryLink(entityLink: EntityLink, type: string) {
        makeEntityLinkPrimary(entityLink, type, this.props.projectKey).then((response) => {
            if (response.status === 200) {
                this.props.onMakePrimary(entityLink);
            }
        });
    }

    openDeleteDialog() {
        this.setState({
            isDeleteLinkModalOpen: true,
        });
    }

    closeDeleteDialog() {
        this.setState({
            isDeleteLinkModalOpen: false,
        });
    }

    render() {
        const { isDeleteLinkModalOpen } = this.state;
        const { type, entityLink, outgoingApplicationName } = this.props;
        return (
            <React.Fragment>
                <DropdownMenu
                    trigger={({ triggerRef, ...props }) => (
                        <Button {...props} testId="entitylinks-droplist" ref={triggerRef}>
                            <MoreIcon size="small" label="" />
                        </Button>
                    )}
                    placement="bottom-end"
                >
                    <DropdownItemGroup>
                        <DropdownItem
                            onClick={() => this.openDeleteDialog()}
                            testId="delete-link-action"
                        >
                            {I18n.getText('entitylinks.more.actions.delete')}
                        </DropdownItem>
                        {!entityLink.isPrimary && (
                            <DropdownItem
                                onClick={() => this.onClickMakePrimaryLink(entityLink, type)}
                                testId="make-primary-link-action"
                            >
                                {I18n.getText('entitylinks.more.actions.make.primary.link')}
                            </DropdownItem>
                        )}
                    </DropdownItemGroup>
                </DropdownMenu>
                <ModalTransition>
                    {isDeleteLinkModalOpen && (
                        <DeleteEntityLinkDialogForm
                            currentApp={this.props.currentApp}
                            outgoingApplicationName={outgoingApplicationName}
                            type={type}
                            projectKey={this.props.projectKey}
                            entityLink={entityLink}
                            onClose={() => this.closeDeleteDialog()}
                            onDelete={this.props.onDelete}
                        />
                    )}
                </ModalTransition>
            </React.Fragment>
        );
    }
}
