/** @typedef {Object} Applink
 * @property {Object} data
 * @property {String} displayUrl
 * @property {String} id
 * @property {String} name
 * @property {Boolean} primary
 * @property {Properties} properties
 * @property {String} rpcUrl
 * @property {Status} status
 * @property {Boolean} statusLoaded
 * @property {Boolean} system
 * @property {"generic"|"bamboo"|"stash"|"confluence"|"refapp"|"fecru"|"jira"} type
 * @property {Function} toJSON
 */

/** @typedef {Object} LegacyApplink
 * @property {"generic"|"bamboo"|"stash"|"confluence"|"refapp"|"fecru"|"jira"} typeId
 * @property {Object[]} hasIncoming
 * @property {Boolean} hasIncoming
 * @property {Boolean} hasOutgoing
 */

/** @typedef {Object} Properties
 * @property {"OAUTH2_PROVIDER" | "OAUTH2_CLIENT" | null} oauth2ConnectionType
 */

/** @typedef {Object} Status
 * @property {Object} localAuthentication
 * @property {{enabled: Boolean}} localAuthentication.outgoing
 * @property {{enabled: Boolean}} localAuthentication.incoming
 */

define('applinks/feature/v3/list', [
    'applinks/lib/lodash',
    'applinks/lib/backbone',
    'applinks/lib/jquery',
    'applinks/lib/window',
    'applinks/lib/aui',
    'applinks/shim/applinks',
    'applinks/common/events',
    'applinks/common/rest',
    'applinks/common/oauth2/rest-client',
    'applinks/common/oauth2/rest-provider',
    'applinks/common/response-status',
    'applinks/feature/status',
    'applinks/feature/status/details',
    'applinks/feature/oauth-dance',
    'applinks/feature/v3/list-views',
    'applinks/feature/v3/ui',
    'applinks/model/applink',
    'applinks/lib/atlassianOauth2Ui',
], function(
    _,
    Backbone,
    $,
    window,
    AJS,
    LegacyApplinks,
    Events,
    ApplinksRest,
    ApplinksOAuth2ClientRest,
    ApplinksOAuth2ProviderRest,
    ResponseStatus,
    ApplinkStatus,
    ApplinkStatusDetails,
    OAuthDance,
    ListViews,
    V3Ui,
    ApplinkModel,
    atlassianOauth2Ui
) {
    var hasLoaded = false;

    var ACTION_DELETE = 'delete';
    var ACTION_LEGACY_EDIT = 'legacy-edit';
    var ACTION_PRIMARY = 'primary';
    var ACTION_REMOTE = 'remote';
    var ACTION_TEST = 'test';
    var ACTION_VIEW_CREDENTIALS = 'view-credentials';
    var ACTION_ATLASSIAN_OAUTH2_VIEW_CREDENTIALS = 'atlassian-oauth2-view-credentials';
    var ACTION_ATLASSIAN_OAUTH2_EDIT_INCOMING = 'atlassian-oauth2-edit-incoming';
    var ACTION_ATLASSIAN_OAUTH2_EDIT_OUTGOING = 'atlassian-oauth2-edit-outgoing';
    var ACTION_RESET_CREDENTIALS = 'reset-credentials';
    var ACTION_ROTATE_CREDENTIALS = 'rotate-credentials';
    var ACTION_REVOKE_CREDENTIALS = 'revoke-credentials';
    var BUILT_IN_ACTIONS = [
        ACTION_DELETE,
        ACTION_LEGACY_EDIT,
        ACTION_PRIMARY,
        ACTION_REMOTE,
        ACTION_TEST,
        ACTION_VIEW_CREDENTIALS,
        ACTION_ATLASSIAN_OAUTH2_VIEW_CREDENTIALS,
        ACTION_ATLASSIAN_OAUTH2_EDIT_INCOMING,
        ACTION_ATLASSIAN_OAUTH2_EDIT_OUTGOING,
        ACTION_RESET_CREDENTIALS,
        ACTION_ROTATE_CREDENTIALS,
        ACTION_REVOKE_CREDENTIALS,
    ];
    var OAUTH2_CLIENT = 'OAUTH2_CLIENT';
    var OAUTH2_PROVIDER = 'OAUTH2_PROVIDER';

    // extra data keys required to render the applinks config screen
    var DATA_KEYS = [
        ApplinkModel.DataKeys.APPLICATION_VERSION,
        ApplinkModel.DataKeys.ATLASSIAN,
        ApplinkModel.DataKeys.CONFIG_URL,
        ApplinkModel.DataKeys.ICON_URI,
        ApplinkModel.DataKeys.EDITABLE,
        ApplinkModel.DataKeys.V3_EDITABLE,
        ApplinkModel.DataKeys.WEB_ITEMS,
        ApplinkModel.DataKeys.ATLASSIAN_OAUTH2_CONFIGURED,
        ApplinkModel.DataKeys.ATLASSIAN_OAUTH2_CREATION_STATUS,
        ApplinkModel.DataKeys.CLIENT_CONFIGURATION_ENTITY_ID,
        ApplinkModel.DataKeys.CLIENT_ENTITY_ID,
        ApplinkModel.DataKeys.EXPIRY_DATE,
        ApplinkModel.DataKeys.SOON_EXPIRES,
        ApplinkModel.DataKeys.CLIENT_ID,
        ApplinkModel.DataKeys.ROTATED_CLIENT_ID,
    ];

    /**
     * Convert to legacy applink format for legacy editing
     *
     * @param {Applink} applink - applink to convert
     * @returns {LegacyApplink & Applink}
     * @private
     */
    function toLegacyApplink(applink) {
        // self link to legacy REST URL
        applink.link = [
            {
                rel: 'self',
                href: ApplinksRest.V1.applink(applink.id).getUrl(),
            },
        ];

        // typeId has changed to type
        applink.typeId = applink.type;
        // mark incoming & outgoing as available
        applink.hasIncoming = true;
        applink.hasOutgoing = true;

        return applink;
    }

    /**
       @param {HTMLElement} element
     * @return {String | undefined}
     */
    function findApplinkRowId(element) {
        return $(element)
            .parents('tr:first')
            .attr('id');
    }

    /**
     * Main table backbone view, handles data
     * loading, remove and updating.
     */
    return Backbone.View.extend({
        template: applinks.feature.v3.list.templates,

        events: {
            'click .legacy-edit-button': 'onLegacyEditClick',
            'click .atlassian-oauth2-edit-button': 'onEditOAuth2',
        },

        initialize: function(options) {
            // extend with options
            _.extend(this, options);
            this.applinks =
                options.model || new ApplinkModel.Collection(null, { dataKeys: DATA_KEYS });
            this.$el.html(
                this.template.table({
                    isV4: V3Ui.isV4(),
                })
            );
            this.registerEvents();

            // initial data load
            this.fetchAllApplinks();
        },

        remove: function() {
            Backbone.View.prototype.remove.call(this);
            this.unregisterEvents();
        },

        /**
         * "Manually" registered events for model and view.
         */
        registerEvents: function() {
            // model events
            // currently only status changes asynchronously, so no need to refresh the view for any other event
            this.listenTo(this.applinks, 'remove destroy', this.onModelRemove).listenTo(
                this.applinks,
                'change:status',
                this.onStatusChange
            );

            // view events
            // actions dropdown callbacks
            $(document).on(
                'click',
                '.v3-applink-actions-dropdown li',
                _.bind(this.onActionsDropdownClick, this)
            );
            // global OAuth dance callback for all authorise links (see applink-status)
            new OAuthDance(this.$el, ApplinkStatus.AUTHENTICATE_LINK_SELECTOR)
                .onSuccess(_.bind(this.onOAuthDanceSuccess, this))
                .initialize();
            // re-fetch all applinks on orphaned upgrade, as it likely results in a new applink
            Events.on(Events.ORPHANED_UPGRADE, this.fetchAllApplinks, this);

            Events.on(Events.ATLASSIAN_OAUTH2_INCOMING_LINK_CREATED, this.fetchAllApplinks, this);
            Events.on(Events.ATLASSIAN_OAUTH2_OUTGOING_LINK_CREATED, this.fetchAllApplinks, this);
            Events.on(Events.ATLASSIAN_OAUTH2_DIALOG_CLOSED, this.fetchAllApplinks, this);
        },

        /**
         * Unregister "manually" registered events outside of `this.$el`.
         */
        unregisterEvents: function() {
            $(document).off('click', '.v3-applink-actions-dropdown li');
            Events.off(Events.ORPHANED_UPGRADE);
        },

        /**
         * Fetch and re-render all applinks.
         */
        fetchAllApplinks: function() {
            this.getTable().removeClass('applinks-loaded');
            this.applinks
                .fetch({ reset: true })
                .always(
                    _.bind(this.filterAndSort, this),
                    _.bind(this.onCollectionUpdate, this, true)
                );
        },

        /**
         * Filter the applinks collection, skipping non-Atlassian system links and moving the built-in (Atlassian)
         * system links to the bottom. System links are generally not important for the user as there's not much that
         * can be done with them.
         */
        filterAndSort: function() {
            // Backbone uses Underscore's stable sortBy so the order of applinks should otherwise be preserved
            var sorted = this.applinks
                .chain()
                .filter(function(applink) {
                    return (
                        !applink.get('system') || applink.getData(ApplinkModel.DataKeys.ATLASSIAN)
                    );
                })
                .sortBy(function(applink) {
                    return applink.get('system') ? 1 : 0;
                })
                .value();
            this.applinks.reset(sorted);
        },

        /**
         * @param {Applink} model - Backbone Model of an Applink
         */
        onStatusChange: function(model) {
            this.onDirectionChange(model.toJSON());

            var statusContainer = this.$el.find('#' + model.id + ' td.status');
            statusContainer.empty();

            // render status
            new ApplinkStatus.View({
                applink: model.toJSON(),
                container: statusContainer,
            });
            // render status details dialog
            new ApplinkStatusDetails.Dialog(model);

            $(document).trigger(Events.APPLINKS_UPDATED);
        },

        /**
         * @param {Applink} model - Backbone Model of an Applink
         */
        onDirectionChange: function(model) {
            if (V3Ui.isV4()) {
                const directionContainer = $('#' + 'agent-table')
                    .find('tr#' + model.id)
                    .find('td:eq(3)')
                    .html('');
                const status = model.status;
                const oauth2ConnectionType = model.properties.oauth2ConnectionType;

                if (status === null) {
                    return;
                }

                if (oauth2ConnectionType === 'OAUTH2_CLIENT' || this.isOutgoingLink(status)) {
                    directionContainer.html(AJS.I18n.getText('applinks.v4.direction.outgoing'));
                } else if (this.isBidirectionalLink(status)) {
                    directionContainer.html(AJS.I18n.getText('applinks.v4.direction.twoway'));
                } else {
                    directionContainer.html(AJS.I18n.getText('applinks.v4.direction.incoming'));
                }
            }
        },

        /**
         * @param {Applink} model - Model of an Applink
         */
        onModelRemove: function(model) {
            ApplinkStatusDetails.removeDialog(model.id);
            if (this.applinks.isEmpty()) {
                this.onCollectionUpdate();
            } else {
                var $container = this.$el.find('tbody');
                this.removeTooltips($container);
                this.$el.find('#' + model.id).remove();
                $(document).trigger(Events.APPLINKS_UPDATED);
                this.renderTooltips($container);
            }
        },

        /**
         * Executed when the Applinks collection is updated.
         *
         * @param {Boolean} isFetched - if the update is resulting from a remote fetch of the applinks list
         */
        onCollectionUpdate: function(isFetched) {
            this.populateTable();
            if (isFetched) {
                // init status fetch for all applinks
                this.fetchStatus();
            }
            this.getTable()
                .find('.in-progress .progress-spinner')
                .spin();
            this.getTable().addClass('applinks-loaded');

            if (hasLoaded) {
                $(document).trigger(Events.APPLINKS_UPDATED);
            } else {
                // Trigger this event once only
                hasLoaded = true;
                $(document).trigger(Events.APPLINKS_LOADED);
            }
        },

        populateTable: function() {
            var rows = document.createDocumentFragment(),
                table = this.getTable(),
                container = table.find('tbody'),
                noApplinksNotice = this.getNoApplinksNotice();

            this.removeTooltips(container);

            this.applinks.each(function(applink) {
                rows.appendChild(new ListViews.Row({ model: applink }).render().get(0));
            });

            // clear container
            container.empty();

            // Update table content (in one go)
            if (rows.childNodes.length > 0) {
                // Show the table and hide the No Applinks notice
                container.append(rows);
                table.removeAttr('data-empty');
                table.prop('hidden', false);
                noApplinksNotice.prop('hidden', true);
            } else {
                // Hide the table and show the No Applinks notice
                table.attr('data-empty', 'empty');
                table.prop('hidden', true);
                noApplinksNotice.prop('hidden', false);
            }

            this.renderTooltips(container);
        },

        /**
         * OAuth dance success callback: refresh status of the link corresponding to the event source.
         */
        onOAuthDanceSuccess: function(source) {
            var applink = this.applinks.get(findApplinkRowId(source));
            applink && applink.fetchStatus();
        },

        /**
         * Actions drop-down menu click handler.
         * @param {MouseEvent} event
         */
        onActionsDropdownClick: function(event) {
            var $element = $(event.target);
            var action = $element.data('action');

            if (action && (_.includes ? _.includes : _.contains)(BUILT_IN_ACTIONS, action)) {
                event.preventDefault();
                var id = $element.parents('.aui-dropdown2:first').data('id');
                var applink, rotatedClientId;
                // trigger selected action
                switch (action) {
                    case ACTION_DELETE:
                        this.confirmDelete(id);
                        break;
                    case ACTION_LEGACY_EDIT:
                        this.legacyEdit(id);
                        break;
                    case ACTION_PRIMARY:
                        this.makePrimary(id);
                        break;
                    case ACTION_REMOTE:
                        this.goToRemote(id);
                        break;
                    case ACTION_VIEW_CREDENTIALS:
                        this.viewOAuth2ProviderCredentials(id);
                        break;
                    case ACTION_TEST:
                        this.testOauth2ClientConnection(id);
                        break;
                    case ACTION_ATLASSIAN_OAUTH2_EDIT_INCOMING:
                        this.editIncomingAtlassianOAuth2Link(id);
                        break;
                    case ACTION_ATLASSIAN_OAUTH2_EDIT_OUTGOING:
                        this.editOutgoingAtlassianOAuth2Link(id);
                        break;
                    case ACTION_ATLASSIAN_OAUTH2_VIEW_CREDENTIALS:
                        this.viewCredentialsAtlassianOAuth2Link(id);
                        break;
                    case ACTION_RESET_CREDENTIALS:
                        this.resetCredentials(id);
                        break;
                    case ACTION_ROTATE_CREDENTIALS:
                        applink = this.applinks.get(id);
                        rotatedClientId = applink && applink.get('data') && applink.get('data')[ApplinkModel.DataKeys.ROTATED_CLIENT_ID];
                        if (rotatedClientId != null) {
                            return;
                        }
                        this.rotateCredentials(id);
                        break;
                    case ACTION_REVOKE_CREDENTIALS:
                        applink = this.applinks.get(id);
                        rotatedClientId = applink && applink.get('data') && applink.get('data')[ApplinkModel.DataKeys.ROTATED_CLIENT_ID];
                        if (rotatedClientId == null) {
                            return;
                        }
                        this.revokeCredentials(id);
                        break;
                }
            }
        },

        /**
         * @param {MouseEvent} event
         */
        onLegacyEditClick: function(event) {
            var id = findApplinkRowId(event.target);
            this.legacyEdit(id);
        },

        /**
         * Show delete confirmation dialog.
         * @param {string} id - Id of {@link Applink}
         */
        confirmDelete: function(id) {
            var dialog = new ListViews.ConfirmDialog({
                id: id,
            });
            dialog.on('confirm', this.deleteApplink, this);

            dialog.show();
        },

        /**
         * Delete Applink
         * @param {string} id - Id of {@link Applink}
         */
        deleteApplink: function(id) {
            // REST Delete call must be synchronous otherwise the applink in the UI (model) will be deleted even if the
            // REST call fails.
            var applink = this.applinks.get(id);
            if (
                applink.attributes &&
                applink.attributes.properties &&
                applink.attributes.properties.oauth2ConnectionType === OAUTH2_CLIENT
            ) {
                const data = ApplinksOAuth2ClientRest.V1.config(id);
                $.ajax({
                    url: data._url,
                    type: 'DELETE',
                    success: function() {
                        applink.collection.remove(applink.cid);
                    },
                });
            } else if (
                applink.attributes &&
                applink.attributes.properties &&
                applink.attributes.properties.oauth2ConnectionType === OAUTH2_PROVIDER
            ) {
                const data = ApplinksOAuth2ProviderRest.V1.client(id);
                $.ajax({
                    url: data._url,
                    type: 'DELETE',
                    success: function() {
                        applink.collection.remove(applink.cid);
                    },
                });
            } else {
                applink.destroy({
                    wait: true,
                    expectedStatuses: [ResponseStatus.Family.SUCCESSFUL, ResponseStatus.NOT_FOUND],
                });
            }
        },

        /**
         * Mark selected application link as primary.
         * @param {string} id - Id of {@link Applink}
         */
        makePrimary: function(id) {
            var self = this;
            this.applinks.get(id).makePrimary({
                success: _.bind(self.fetchAllApplinks, self),
            });
        },

        /**
         * Open the legacy edit dialog for a given applink `id`.
         * If the applink is an oauth2 connection it will redirect to the correct OAuth 2 screen
         *
         * @param {string} id - Id of {@link Applink}
         */
        legacyEdit: function(id) {
            LegacyApplinks.UI.hideInfoBox();

            var applink = this.applinks.get(id).toJSON();
            if (applink.properties && applink.properties.oauth2ConnectionType === OAUTH2_PROVIDER) {
                window.location.href =
                    AJS.contextPath() + '/plugins/servlet/oauth2/provider?id=' + applink.id;
                return;
            } else if (
                applink.properties &&
                applink.properties.oauth2ConnectionType === OAUTH2_CLIENT
            ) {
                window.location.href =
                    AJS.contextPath() + '/plugins/servlet/oauth2/client?clientId=' + applink.id;
                return;
            }

            var legacyApplink = toLegacyApplink(applink);
            var refreshApplinks = _.bind(this.fetchAllApplinks, this);

            LegacyApplinks.editAppLink(
                legacyApplink,
                'undefined',
                false,
                refreshApplinks,
                refreshApplinks
            );
        },

        /**
         * Open remote in new window.
         * @param {string} id - Id of {@link Applink}
         */
        goToRemote: function(id) {
            var applink = this.applinks.get(id).toJSON();
            if (
                applink.properties.oauth2ConnectionType === OAUTH2_PROVIDER ||
                applink.properties.oauth2ConnectionType === OAUTH2_CLIENT
            ) {
                window.open(applink.displayUrl);
            } else {
                window.open(this.applinks.get(id).getAdminUrl());
            }
        },

        /**
         * @param {string} id - Id of {@link Applink}
         */
        testOauth2ClientConnection: function(id) {
            var applink = this.applinks.get(id).toJSON();
            var isOAuth2Client =
                applink.properties && applink.properties.oauth2ConnectionType === OAUTH2_CLIENT;
            var isAtlassianOAuth2Link = applink.data && applink.data.oauth2Configured;
            if (isOAuth2Client || isAtlassianOAuth2Link) {
                var flowTestId = isAtlassianOAuth2Link
                    ? applink.data[ApplinkModel.DataKeys.CLIENT_CONFIGURATION_ENTITY_ID]
                    : applink.id;
                window.location.href =
                    AJS.contextPath() + '/plugins/servlet/oauth2/client?flowTest=' + flowTestId;
            }
        },

        /**
         * View Client Credentials for OAuth 2.0 Provider connection
         * @param {string} id - Id of {@link Applink}
         */
        viewOAuth2ProviderCredentials: function(id) {
            var applink = this.applinks.get(id).toJSON();
            if (applink.properties && applink.properties.oauth2ConnectionType === OAUTH2_PROVIDER) {
                window.location.href =
                    AJS.contextPath() +
                    '/plugins/servlet/oauth2/provider?id=' +
                    applink.id +
                    '&viewCredentials=true';
            }
        },

        /**
         * Init fetch Applink status for all applinks.
         */
        fetchStatus: function() {
            // init fetch status on all applinks,
            this.applinks.each(function(applink) {
                applink.fetchStatus();
            });
        },

        /**
         * Initialize tipsy handler.
         */
        renderTooltips: function($element) {
            // first remove any existing tipsy element
            $('body > .tipsy').remove();

            // re-initialize tipsy, search only for data-title
            ($element || this.$el).find('[data-title]').tooltip({
                className: 'tipsy-left',
                title: 'data-title',
                html: true,
            });

            ($element || this.$el).find('span.applinks-name').each(function() {
                const displayUrl = this.getAttribute('data-displayUrl');
                const rpcUrl = this.getAttribute('data-rpcUrl');
                if (displayUrl || rpcUrl) {
                    $(this).tooltip({
                        className: 'url-tipsy',
                        gravity: 's',
                        html: true,
                        title: function() {
                            return displayUrl + '<br/>' + rpcUrl;
                        },
                    });
                }
            });
        },

        removeTooltips: function($element) {
            // re-initialize tipsy, search only for data-title
            ($element || this.$el).find('[data-title]').attr('data-title', '');
        },

        getTable: function() {
            return this.$el.find('#agent-table');
        },

        getNoApplinksNotice: function() {
            return this.$el.find('#v3-no-applinks-notice');
        },

        /**
         * @param {Status} status
         * @return {Boolean}
         */
        isBidirectionalLink: function(status) {
            return (
                status.localAuthentication.incoming.enabled &&
                status.localAuthentication.outgoing.enabled
            );
        },

        /**
         * @param {Status} status
         * @return {Boolean}
         */
        isOutgoingLink: function(status) {
            return (
                status.localAuthentication.outgoing.enabled &&
                !status.localAuthentication.incoming.enabled
            );
        },

        onEditOAuth2: function(event) {
            const id = $(event.target)
                .parents('tr')
                .attr('id');
            this.editIncomingAtlassianOAuth2Link(id);
        },

        editIncomingAtlassianOAuth2Link: function(id) {
            atlassianOauth2Ui.openCreateIncomingLinkDialog({ applinkId: id });
        },

        editOutgoingAtlassianOAuth2Link: function(id) {
            atlassianOauth2Ui.openCreateOutgoingLinkDialog({ applinkId: id });
        },

        viewCredentialsAtlassianOAuth2Link: function(id) {
            atlassianOauth2Ui.openIncomingLinkDetailsDialog({ applinkId: id });
        },

        resetCredentials: function(id) {
            const applink = this.applinks.get(id);
            const clientId = applink && applink.get('data') && applink.get('data')[ApplinkModel.DataKeys.CLIENT_ID];
            atlassianOauth2Ui.openResetCredentialsDialog({
                clientId: clientId,
                applinkId: id
            });
        },

        rotateCredentials: function(id) {
            const applink = this.applinks.get(id);
            const expiryDate = applink && applink.get('data') && applink.get('data')[ApplinkModel.DataKeys.EXPIRY_DATE];
            const clientId = applink && applink.get('data') && applink.get('data')[ApplinkModel.DataKeys.CLIENT_ID];
            atlassianOauth2Ui.openRotateCredentialsDialog({
                expiryDate: expiryDate,
                clientId: clientId,
                applinkId: id
            });
        },
        revokeCredentials: function(id) {
            const applink = this.applinks.get(id);
            const rotatedClientId = applink && applink.get('data') && applink.get('data')[ApplinkModel.DataKeys.ROTATED_CLIENT_ID];
            const clientEntityId = applink && applink.get('data') && applink.get('data')[ApplinkModel.DataKeys.CLIENT_ENTITY_ID];
            atlassianOauth2Ui.openRevokeCredentialsDialog({
                rotatedClientId: rotatedClientId,
                clientEntityId: clientEntityId
            });
        },
    });
});
