require([
    "jquery",
    "troubleshooting/ajs",
    "aui/flag",
    'troubleshooting/user-utils'
], function ($, AJS, flag, UserUtils) {

    AJS.toInit(function () {
        const user = UserUtils.getCurrentUser();
        const settingsUrl = AJS.contextPath() + '/rest/troubleshooting/1.0/user-setting/' + user;
        const severityUrl = AJS.contextPath() + '/rest/troubleshooting/1.0/user-setting/' + user + '/notification-severity';
        const watchUrl = AJS.contextPath() + '/rest/troubleshooting/1.0/user-setting/' + user + '/watch';

        const containerDeferred = $.Deferred();
        const preferencesDeferred = $.getJSON(settingsUrl);

        // Load the notification config panel once the troubleshootingContainerLoaded event is triggered
        // AND the UserPreferences REST call has returned.
        AJS.bind("troubleshootingContainerLoaded", containerDeferred.resolve);

        $.when(containerDeferred, preferencesDeferred).done(function(containerResult, preferencesData) {
            renderConfigPanel(containerResult, preferencesData, severityUrl, watchUrl);
        });
    });

    // containerResult is just used for padding with the $.when callback.
    function renderConfigPanel(containerResult, [preferences], severityUrl, watchUrl) {
        var { severityThresholdForNotifications } = preferences;
        var notificationConfig = $(Support.HealthCheck.Templates.notificationConfig(preferences));

        var watchCheckbox = notificationConfig.find('#notification-watch');
        var severitySelect = notificationConfig.find('select#notification-settings');

        watchCheckbox.change(function () { updateWatch.call(this, watchUrl); });
        severitySelect.val(severityThresholdForNotifications.toLowerCase());
        severitySelect.change(function () { updateSeverity.call(this, severityUrl); });
        $('.troubleshooting-status-container').after(notificationConfig);
    }

    // Make a REST call to watch/unwatch for current user
    function updateWatch(watchUrl) {
        const checked = $(this).is(":checked");
        $.ajax({
            url: watchUrl,
            type: checked ? 'PUT' : 'DELETE',
            contentType: 'application/json'
        })
            .fail(function (e) {
                console.error('Something went wrong, could not update watch status', e);
                showErrorFlag();
            });

        AJS.trigger('analyticsEvent', {
            name: `healthcheck.notification.${checked ? 'watch' : 'unwatch'}`,
        });
    }

    // Make a REST call to enable or disable notification for current user
    function updateSeverity(severityUrl) {
        // Select option corresponds to Severity API Model enum - pass straight through to backend.
        var notificationSeverity = $(this).val();
        $.ajax({
            url: severityUrl,
            type: 'PUT',
            data: JSON.stringify(notificationSeverity),
            contentType: 'application/json'
        })
            .done(function () {
                // NOOP
            })
            .fail(function (e) {
                console.error('Something went wrong, could not save the notification preference', e);
                showErrorFlag();
            });

        // Trigger Atlassian Analytics event after user updated their settings
        // Available options: [undefined, major, critical], critical is for disabling all notification
        AJS.trigger('analyticsEvent', {
            name: "healthcheck.notification.config.update." + notificationSeverity,
            data: {severity: notificationSeverity}
        });
    }

    function showErrorFlag() {
        flag({type: 'error', body: AJS.I18n.getText('stp.health.settings.update.error') })
    }
});