'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

define('bitbucket-plugin-default-reviewers/internal/feature/default-reviewers/dialog/default-reviewers-dialog', ['@atlassian/aui', 'jquery', 'lodash', 'eve', 'bitbucket/util/navbuilder', 'bitbucket/util/server', 'bitbucket/util/state', 'bitbucket/internal/feature/user/user-multi-selector/user-multi-selector', 'bitbucket/internal/model/page-state', 'bitbucket/internal/enums', 'bitbucket/internal/util/scope-type', 'bitbucket-plugin-default-reviewers/internal/model/default-reviewers/ref-matcher'], function (AJS, $, _, eve, nav, server, pageState, UserMultiSelector, modelPageState, _ref, scopeTypeUtil, RefMatcher) {
    var ScopeType = _ref.ScopeType;

    /**
     * Enum for the available pickers
     * @type {{ANY_REF: string, BRANCH: string, PATTERN: string, MODEL: string}}
     */
    var PickerTypes = {
        ANY_REF: 'ANY_REF',
        BRANCH: 'BRANCH',
        PATTERN: 'PATTERN',
        MODEL: 'MODEL'
    };

    var MatcherSourceTypes = {
        SOURCE: 'source',
        TARGET: 'target'
    };

    /**
     *
     * @param {Object} options
     * @param {Object?} options.dialogOptions
     * @param {DefaultReviewerSet?} options.defaultReviewerSet
     * @param {boolean?} options.dialogOptions.defaultBranch - the repository's default branch
     * @param {boolean?} options.dialogOptions.isEditing - are we editing?
     * @param {boolean?} options.dialogOptions.callback - called when saving
     * @param {ScopeType} options.scopeType
     * @constructor
     */
    function DefaultReviewersDialog() {
        var options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

        _.bindAll(this, _.functions(this.constructor.prototype));
        this._scopeType = options.scopeType;
        this.currentScopeNavMethod = scopeTypeUtil.scopeNavAndStateMethod(this._scopeType);
        this._dialogOptions = options.dialogOptions || {};
        this.defaultReviewerSet = options.defaultReviewerSet;
        this.callback = this._dialogOptions.callback || _.noop;
        this._dialogOptions.defaultReviewerSet = this.defaultReviewerSet && this.defaultReviewerSet.toJSON(); // used in soy template
        this._dialogOptions.scopeType = this._scopeType; // used in soy template
        this._dialog = AJS.dialog2(bitbucketPluginDefaultReviewers.internal.feature.defaultReviewers.dialog.defaultReviewersDialog.defaultReviewersDialog(this._dialogOptions));
        this._dialog.$el.on('change', 'input, #default-reviewers-selector-trigger, select', this.validate);
    }

    DefaultReviewersDialog.prototype.show = function () {
        this._dialog.show();
        this.init();
    };

    DefaultReviewersDialog.prototype.init = function () {
        getBranchModels(this.currentScopeNavMethod, this._scopeType).fail(this.disableBranchModel);
        this.initMatcherTypeSelectors();
        this.initDefaultReviewersSelector();
        this.initButtons();
        this.validate();
    };

    DefaultReviewersDialog.prototype.initMatcherTypeSelectors = function () {
        var _this = this;

        var updateBranchField = function updateBranchField(val, matcherSourceType) {
            _this.updateBranchField({
                idPrefix: matcherSourceType + '-branch',
                branchType: val,
                defaultBranch: matcherSourceType === MatcherSourceTypes.TARGET ? _this._dialogOptions.defaultBranch : undefined
            });
        };

        Object.keys(MatcherSourceTypes).map(function (x) {
            return MatcherSourceTypes[x];
        }).forEach(function (matcherSourceType) {
            var $matcherType = $('input[name="' + matcherSourceType + '-branch-matcherType"]');

            if (_this._dialogOptions.isEditing) {
                $matcherType.parents('.radio').remove();

                return;
            }

            updateBranchField($matcherType.filter(':checked').val(), matcherSourceType);

            $matcherType.on('change', function (e) {
                _this.validate();
                updateBranchField(e.target.value, matcherSourceType);
                e.target.focus();
            });
        });
    };

    /**
     *
     * @param {string} idPrefix - the branch model selector id prefix
     */
    DefaultReviewersDialog.prototype.initBranchModelSelector = function (idPrefix) {
        /**
         * Return a string containing the HTML to be used for the custom select row
         * @param {Select2Item} item - a select2 row item
         * @returns {string}
         */
        function tplFunction(item) {
            var $optEl = $(item.element);

            return bitbucketPluginDefaultReviewers.internal.feature.defaultReviewers.dialog.defaultReviewersDialog.branchModelInputRowTemplate({
                name: $optEl.text(),
                pattern: $optEl.data('display-name'),
                isBranch: $optEl.data('is-branch') === true
            });
        }

        var $branchModelSelector = $('#' + idPrefix + '-branch-model-selector');
        $branchModelSelector.auiSelect2({
            placeholder: AJS.I18n.getText('bitbucket.defaultreviewers.settings.dialog.branchmodelselector.placeholder'),
            formatResult: tplFunction,
            formatSelection: tplFunction,
            // Ensures the search box is hidden for this selector
            minimumResultsForSearch: Infinity
        });
        $branchModelSelector.on('change', this.validate);
    };

    DefaultReviewersDialog.prototype.initDefaultReviewersSelector = function () {
        // add user selector params to filter out unlicensed/unauthorized users
        var userSelectorOptions = {
            urlParams: {
                'permission.1': 'LICENSED_USER' // only licensed users
            },
            initialItems: this.defaultReviewerSet ? this.defaultReviewerSet.getReviewers().concat(this.defaultReviewerSet.getReviewerGroups()) : []
        };
        var project = modelPageState.getProject().getKey();

        if (this._scopeType === ScopeType.REPOSITORY) {
            var repo = modelPageState.getRepository().getName();
            userSelectorOptions.url = nav.rest('ui').project(project).repo(repo).addPathComponents('reviewer-groups').build();
            userSelectorOptions.urlParams['permission.2'] = 'REPO_READ'; // only users that can read the repository
            userSelectorOptions.urlParams['permission.2.repositoryId'] = modelPageState.getRepository().getId();
        } else {
            userSelectorOptions.url = nav.rest('ui').project(project).addPathComponents('reviewer-groups').build();
            userSelectorOptions.urlParams['permission.2'] = 'PROJECT_READ'; // only users that can read the project
            userSelectorOptions.urlParams['permission.2.projectId'] = modelPageState.getProject().getId();
        }

        var $defaultReviewersSelector = $('#default-reviewers-selector');
        this.defaultReviewersSelector = new UserMultiSelector($defaultReviewersSelector, userSelectorOptions);
        $defaultReviewersSelector.on('change', this.updateApprovalsRequiredDropdown);
        this.updateApprovalsRequiredDropdown();
    };

    DefaultReviewersDialog.prototype.updateApprovalsRequiredDropdown = function () {
        var reviewerUsers = new Set();

        this.defaultReviewersSelector.getSelectedItems().forEach(function (item) {
            if (item.type === 'REVIEWER_GROUP') {
                item.users.forEach(function (user) {
                    if (user.active) {
                        reviewerUsers.add(user.name);
                    }
                });
            } else if (item.active) {
                reviewerUsers.add(item.name);
            }
        });

        var $requiredApproverCountSelector = $('#required-approver-count-selector');
        var selectedVal = this.defaultReviewerSet && this.defaultReviewerSet.getRequiredApprovals() || $requiredApproverCountSelector.val();

        $requiredApproverCountSelector.empty();

        var options = '';
        _.range(0, reviewerUsers.size + 1).forEach(function (val) {
            options += '<option value="' + val + '">' + val + '</option>';
        });
        $requiredApproverCountSelector.append(options);

        // Select the existing value if possible
        if ($requiredApproverCountSelector.find('option[value=' + selectedVal + ']').length) {
            $requiredApproverCountSelector.val(selectedVal);
        }
    };

    /**
     * Attaches the event handlers to the save and cancel buttons in the Dialog's footer
     */
    DefaultReviewersDialog.prototype.initButtons = function () {
        $('#save-default-reviewer-button').on('click', this.save);
        $('#cancel-default-reviewer-button').on('click', this.hide);
    };

    /**
     * Get the source RefMatcher of either the current DefaultReviewerSet or the selected value from the UI
     * @returns {RefMatcher}
     */
    DefaultReviewersDialog.prototype.getCurrentFromMatcher = function () {
        return this.defaultReviewerSet && this.defaultReviewerSet.getSourceRefMatcher() || this.getCurrentMatcher(MatcherSourceTypes.SOURCE);
    };

    /**
     * Get the target RefMatcher of either the current DefaultReviewerSet or the selected value from the UI
     * @returns {RefMatcher}
     */
    DefaultReviewersDialog.prototype.getCurrentToMatcher = function () {
        return this.defaultReviewerSet && this.defaultReviewerSet.getTargetRefMatcher() || this.getCurrentMatcher(MatcherSourceTypes.TARGET);
    };

    /**
     * Get the current matcher
     * @param {string} origin - the ref origin, 'source' or 'target'
     * @returns {?RefMatcher}
     */
    DefaultReviewersDialog.prototype.getCurrentMatcher = function (origin) {
        switch ($('#' + origin + '-branch-matcherType').find('input:checked').val()) {
            case PickerTypes.BRANCH:
                var selectedRef = null;

                if (this._scopeType === ScopeType.REPOSITORY) {
                    selectedRef = $('#' + origin + '-branch-branch-selector-trigger').find('.name').data('revision-ref');
                } else {
                    var branchVal = $('#' + origin + '-branch-branch-name-field').val();

                    if (branchVal && !_.includes(branchVal, ' ')) {
                        selectedRef = {
                            id: branchVal,
                            displayId: branchVal
                        };
                    }
                }

                return selectedRef ? new RefMatcher().branch(selectedRef) : null;
            case PickerTypes.PATTERN:
                return new RefMatcher().pattern($('#' + origin + '-branch-branch-pattern').val());
            case PickerTypes.MODEL:
                var $branchModelSelector = $('#' + origin + '-branch-branch-model-selector');
                var branchModelFieldValue = $branchModelSelector.val();
                var branchModelFieldValueIsBranch = $branchModelSelector.find(':selected').data('is-branch');

                return new RefMatcher().model(branchModelFieldValue, branchModelFieldValueIsBranch);
            case PickerTypes.ANY_REF:
                return new RefMatcher().any();
        }
    };

    /**
     * Removes all errors from the dialog.
     */
    DefaultReviewersDialog.prototype.resetErrors = function () {
        this._dialog.$el.find('.error').remove();
    };

    /**
     * Renders errors to the top of the dialog content.
     * @param {Array<Object>} containing errors to render
     */
    DefaultReviewersDialog.prototype.renderErrors = function (errors) {
        this.resetErrors();
        if (errors.responseJSON) {
            errors = errors.responseJSON.errors;
        }
        var errorHTMLMessages = [];
        var unknownError = AJS.I18n.getText('bitbucket.defaultreviewers.error.unknown');

        if (!Array.isArray(errors) || !errors.filter(function (error) {
            return error.message;
        })) {
            errorHTMLMessages.push(aui.message.error({ content: unknownError }));
        } else {
            errorHTMLMessages = errors.map(function (error) {
                return aui.message.error({ content: _.escape(error.message) });
            });
        }

        $(errorHTMLMessages.join('')).insertBefore($('.default-reviewers-dialog-form'));
    };

    /**
     * Renders any errors returned in the AJAX response
     * @param response
     * @param textStatus
     * @param errorThrown
     * @param data The returned content
     * @returns {boolean}
     */
    DefaultReviewersDialog.prototype.responseErrorHandler = function (response, textStatus, errorThrown, data) {
        if (data.errors && data.errors.length) {
            this.renderErrors(data.errors);

            return false;
        }
    };

    DefaultReviewersDialog.prototype.save = function () {
        var _this2 = this;

        this.showSpinner();
        if (!this.isValid()) {
            this.stopSpinner();

            return;
        }

        this.disableSubmit(true);

        var sourceMatcher = this.getCurrentFromMatcher();
        var targetMatcher = this.getCurrentToMatcher();

        var urlBuilder = nav.rest('default-reviewers', 'latest')[this.currentScopeNavMethod.nav](pageState[this.currentScopeNavMethod.state]()).addPathComponents('condition').withParams({ avatarSize: 32 }).addPathComponents(this.defaultReviewerSet ? this.defaultReviewerSet.getId() : '');

        // Separate reviewers and reviewer groups
        var reviewers = [];
        var reviewerGroups = [];
        this.defaultReviewersSelector.getSelectedItems().forEach(function (item) {
            if (item.type === 'REVIEWER_GROUP') {
                reviewerGroups.push(item);
            } else {
                reviewers.push(item);
            }
        });

        server.rest({
            contentType: 'application/json',
            url: urlBuilder.build(),
            data: {
                sourceMatcher: {
                    id: sourceMatcher.id,
                    type: {
                        id: sourceMatcher.getType().id,
                        name: sourceMatcher.getType().name
                    },
                    displayId: sourceMatcher.displayId
                },
                targetMatcher: {
                    id: targetMatcher.id,
                    type: {
                        id: targetMatcher.getType().id,
                        name: targetMatcher.getType().name
                    },
                    displayId: targetMatcher.displayId
                },
                reviewers: reviewers,
                reviewerGroups: reviewerGroups,
                requiredApprovals: $('#required-approver-count-selector').val()
            },
            type: this.defaultReviewerSet ? 'PUT' : 'POST',
            statusCode: {
                400: this.responseErrorHandler,
                404: this.responseErrorHandler,
                500: this.responseErrorHandler
            }
        }).then(this.callback).then(this.hide).then(function () {
            var title = AJS.I18n.getText('bitbucket.defaultreviewers.settings.added.success');

            if (_this2._dialogOptions.isEditing) {
                title = AJS.I18n.getText('bitbucket.defaultreviewers.settings.edited.success');
            }
            AJS.flag({
                type: 'success',
                title: title,
                persistent: false,
                close: 'auto',
                body: ''
            });
        }).always(function () {
            _this2.stopSpinner();
            _this2.disableSubmit(false);
        });
    };

    /**
     * Validates the form in the dialog.
     * If the form is valid, enables the submit button, if not disables it.
     */
    DefaultReviewersDialog.prototype.validate = function () {
        this.disableSubmit(!this.isValid());
    };

    /**
     * Check if the form, in it's current state is valid.
     * If the form is not valid, disable the submit button.
     * @returns {boolean} true if the form is valid, false otherwise
     */
    DefaultReviewersDialog.prototype.isValid = function () {
        var fromMatcher = this.getCurrentFromMatcher();
        var toMatcher = this.getCurrentToMatcher();

        var hasMatchers = fromMatcher != null && toMatcher != null;
        var validFromMatcherValue = hasMatchers && fromMatcher.getId() !== '';
        var validToMatcherValue = hasMatchers && toMatcher.getId() !== '';
        var validReviewersValue = this.defaultReviewersSelector && this.defaultReviewersSelector.getSelectedItems().length;

        return hasMatchers && validFromMatcherValue && validToMatcherValue && validReviewersValue;
    };

    DefaultReviewersDialog.prototype.hide = function () {
        this._dialog.hide();
    };

    DefaultReviewersDialog.prototype.showSpinner = function () {
        this._dialog.$el.find('.spinner').spin();
    };

    DefaultReviewersDialog.prototype.stopSpinner = function () {
        this._dialog.$el.find('.spinner').spinStop();
    };

    /**
     * Sets the disabled attribute on the submit button
     * @param {boolean} isDisabled to set the disabled attribute to
     */
    DefaultReviewersDialog.prototype.disableSubmit = function (isDisabled) {
        $('#save-default-reviewer-button').prop('disabled', isDisabled);
    };

    DefaultReviewersDialog.prototype.initBranchSelectorInput = function (idPrefix) {
        var branchSelectorInput = $('#' + idPrefix + '-input-field .branch-selector-input')[0];

        if (branchSelectorInput) {
            var _branchSelectorInput$ = branchSelectorInput.dataset,
                id = _branchSelectorInput$.id,
                data = _objectWithoutProperties(_branchSelectorInput$, ['id']);

            eve('bitbucket.internal.DO_NOT_USE.widget.branchselector.inputAdded', null, id, data);
        }
    };

    /**
     * Replaces the branch ref selector with the specified template content.
     * Validates the form, to ensure the submit button is disabled with the new, empty field, being in the form.
     * @param {object} opts
     * @param {string} opts.idPrefix - the prefix for the id of the selector
     * @param {string} opts.branchType - the branch {PickerTypes}
     * @param {string?} opts.defaultBranch - the default branch, if given will be the default value of the branch name selector
     */
    DefaultReviewersDialog.prototype.updateBranchField = function (opts) {
        var updateDOMFields = function updateDOMFields(options) {
            var $inputField = $('#' + options.idPrefix + '-input-field');
            $inputField.html(bitbucketPluginDefaultReviewers.internal.feature.defaultReviewers.dialog.defaultReviewersDialog.branchRefField(_extends({
                scopeType: this._scopeType
            }, options)));
            this.initBranchSelectorInput(options.idPrefix);
            this.initBranchModelSelector(options.idPrefix);
            $inputField.on('input change', this.validate);

            // detach this item from the DOM and move it to the radio that triggered it
            var $callee = $('#' + opts.idPrefix + '-matcherType input:checked').parents('.radio');
            $callee.append($inputField);

            this.validate();
        };

        if (opts.branchType === PickerTypes.MODEL) {
            getBranchModels(this.currentScopeNavMethod, this._scopeType).then(function (branchModels) {
                opts.branchModels = branchModels;
            }).done(updateDOMFields.bind(this, opts));
        } else {
            updateDOMFields.bind(this, opts)();
        }
    };

    /**
     * Removes the "Branch model" select option from the selector
     */
    DefaultReviewersDialog.prototype.disableBranchModel = function () {
        $('#source-branch-matcherType-model, #target-branch-matcherType-model').parents('.radio').remove();
    };

    /**
     * Queries the REST API for the current branch model.
     * @returns {Promise<Array<Object>>}
     */
    var getBranchModels = _.once(function getBranchModels(currentScopeNavMethod, scopeType) {
        var urlBuilder = nav.rest('branch-utils', 'latest')[currentScopeNavMethod.nav](pageState[currentScopeNavMethod.state]()).addPathComponents('branchmodel');

        if (scopeType === ScopeType.PROJECT) {
            urlBuilder = urlBuilder.addPathComponents('configuration');
        }
        var reject = function reject() {
            return $.Deferred().reject();
        };

        return server.rest({
            url: urlBuilder.build(),
            statusCode: {
                // if there is an error getting the branch model config, reject here so it gets disabled as a choice
                400: reject,
                404: reject,
                409: reject
            }
        }).then(function (response) {
            var branchModels = [];

            if (response.development) {
                branchModels.push({
                    id: 'development',
                    name: AJS.I18n.getText('bitbucket.defaultreviewers.settings.dialog.model.development'),
                    pattern: response.development.refId || response.development.displayId || AJS.I18n.getText('bitbucket.defaultreviewers.settings.model.default.branch'),
                    isBranch: true
                });
            }

            if (response.production) {
                branchModels.push({
                    id: 'production',
                    name: AJS.I18n.getText('bitbucket.defaultreviewers.settings.dialog.model.production'),
                    pattern: response.production.refId || response.production.displayId || AJS.I18n.getText('bitbucket.defaultreviewers.settings.model.default.branch'),
                    isBranch: true
                });
            }

            if (response.types) {
                response.types.forEach(function (type) {
                    branchModels.push({
                        id: type.id,
                        name: type.displayName,
                        pattern: type.prefix,
                        isBranch: false
                    });
                });
            }

            return branchModels;
        });
    });

    return DefaultReviewersDialog;
});