'use strict';

var _lodash = require('lodash');

var _ajax = require('bitbucket/internal/util/ajax');

var ajax = _interopRequireWildcard(_ajax);

var _analytics = require('bitbucket/internal/util/analytics');

var analytics = _interopRequireWildcard(_analytics);

var _clientStorage = require('bitbucket/internal/util/client-storage');

var clientStorage = _interopRequireWildcard(_clientStorage);

var _navbuilder = require('bitbucket/util/navbuilder');

var nav = _interopRequireWildcard(_navbuilder);

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) newObj[key] = obj[key]; } } newObj.default = obj; return newObj; } }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; }

var DEFAULT_COLOR_FILTER = 'red-green';
var ALT_DIFF_COLOR_PROP = 'alternateDiffColors';
var HTML_COLOR_FILTER_ATTR = 'data-color-accessibility-mode';
var ANALYTICS_EVENT = 'stash.client.color.filter.user.migrated';

var getColorFilterURL = function getColorFilterURL() {
    return nav.rest('ui').addPathComponents('accessibility', 'color-filter').build();
};

/**
 *  Generates the localStorage key where ALT_DIFF_COLOR_PROP may be present
 */
var buildUserStorageKey = function buildUserStorageKey(data) {
    if (!data || !data.currentUser) {
        return null;
    }

    return 'user-' + data.currentUser.slug + '-diff_options';
};

/**
 *  Verifies if ALT_DIFF_COLOR_PROP exists in diff options
 *  and it is equal to true. returns false otherwise.
 */
var isAlternativeColorDiffEnabled = function isAlternativeColorDiffEnabled(storageKey) {
    if (!storageKey) {
        return false;
    }

    var diffOptionsData = clientStorage.getItem(storageKey);

    return !!(0, _lodash.get)(diffOptionsData, ALT_DIFF_COLOR_PROP);
};

/**
 *  Verifies if HTML_COLOR_FILTER_ATTR is present on the <html> tag
 *  returns false otherwise.
 */
var isColorFilterEnabled = function isColorFilterEnabled() {
    return !!window.document.documentElement.attributes[HTML_COLOR_FILTER_ATTR];
};

/**
 *  Deletes ALT_DIFF_COLOR_PROP from localStorage.
 */
var deleteAlternativeColorDiff = function deleteAlternativeColorDiff(storageKey) {
    if (!storageKey) return;

    var diffOptionsData = clientStorage.getItem(storageKey);

    if (!diffOptionsData) return;

    delete diffOptionsData[ALT_DIFF_COLOR_PROP];

    clientStorage.setItem(storageKey, diffOptionsData);
};

/**
 *  Verifies if the color filter setting exist on the current accounts, if
 *  the endpoint returns 404 then the color filter doesn't exist and we resolve true,
 *  otherwise we'll resolve false.
 *
 *  However in case of any other error we'll reject the promise to alert the
 *  parent process to stop.
 */
var isColorFilterNeverEnabled = function isColorFilterNeverEnabled() {
    return new Promise(function (resolve, reject) {
        ajax.rest({
            url: getColorFilterURL(),
            statusCode: {
                404: function _() {
                    resolve(true);

                    // prevent error dialogue from appearing
                    return false;
                },
                200: function _() {
                    return resolve(false);
                },
                '*': function _() {
                    reject('Unable to verify user settings for migration');

                    // prevent error dialogue from appearing
                    return false;
                }
            }
        }).done(function () {
            return resolve(false);
        });
    });
};

/**
 *  Sets the color filter setting to DEFAULT_COLOR_FILTER for the current account
 *  If an error is thrown the promise will be rejected.
 */
var setDefaultColorFilter = function setDefaultColorFilter(userId) {
    return new Promise(function (resolve, reject) {
        ajax.rest({
            url: getColorFilterURL(),
            type: 'PUT',
            data: {
                colorFilter: DEFAULT_COLOR_FILTER
            },
            statusCode: {
                200: function _() {
                    analytics.add(ANALYTICS_EVENT, {
                        'user.id': userId
                    });

                    resolve();
                },
                '*': function _() {
                    reject('Unable to set user default setting');

                    // prevent error dialogue from appearing
                    return false;
                }
            }
        }).done(resolve);
    });
};

/**
 * This algorithms migrates current users using the old alternative colors
 * as defined in
 * https://hello.atlassian.net/wiki/spaces/BSERV/pages/4547160425/User+migration
 */
var migrate = function () {
    var _ref = _asyncToGenerator( /*#__PURE__*/regeneratorRuntime.mark(function _callee(data) {
        var storageKey;
        return regeneratorRuntime.wrap(function _callee$(_context) {
            while (1) {
                switch (_context.prev = _context.next) {
                    case 0:
                        storageKey = buildUserStorageKey(data);

                        if (!isAlternativeColorDiffEnabled(storageKey)) {
                            _context.next = 15;
                            break;
                        }

                        if (isColorFilterEnabled()) {
                            _context.next = 14;
                            break;
                        }

                        _context.prev = 3;
                        _context.next = 6;
                        return isColorFilterNeverEnabled();

                    case 6:
                        if (!_context.sent) {
                            _context.next = 9;
                            break;
                        }

                        _context.next = 9;
                        return setDefaultColorFilter((0, _lodash.get)(data, 'currentUser.id'));

                    case 9:
                        _context.next = 14;
                        break;

                    case 11:
                        _context.prev = 11;
                        _context.t0 = _context['catch'](3);
                        throw _context.t0;

                    case 14:

                        deleteAlternativeColorDiff(storageKey);

                    case 15:
                    case 'end':
                        return _context.stop();
                }
            }
        }, _callee, undefined, [[3, 11]]);
    }));

    return function migrate(_x) {
        return _ref.apply(this, arguments);
    };
}();

define('@bitbucket/internal/feature/color-filter-migration', ['exports'], function (exports) {
    exports.migrate = migrate;

    // exporting for testing purposes
    exports._getColorFilterURL = getColorFilterURL;
    exports._buildUserStorageKey = buildUserStorageKey;
    exports._isAlternativeColorDiffEnabled = isAlternativeColorDiffEnabled;
    exports._deleteAlternativeColorDiff = deleteAlternativeColorDiff;
    exports._isColorFilterNeverEnabled = isColorFilterNeverEnabled;
    exports._setDefaultColorFilter = setDefaultColorFilter;
});