define('bitbucket/internal/color-filter-migration/color-filter-migration.test', ['bitbucket/internal/util/ajax', 'bitbucket/internal/util/analytics', 'bitbucket/internal/util/client-storage', './color-filter-migration.js'], function (_ajax, _analytics, _clientStorage, _colorFilterMigration) {
    'use strict';

    function _asyncToGenerator(fn) {
        return function () {
            var gen = fn.apply(this, arguments);
            return new Promise(function (resolve, reject) {
                function step(key, arg) {
                    try {
                        var info = gen[key](arg);
                        var value = info.value;
                    } catch (error) {
                        reject(error);
                        return;
                    }

                    if (info.done) {
                        resolve(value);
                    } else {
                        return Promise.resolve(value).then(function (value) {
                            step("next", value);
                        }, function (err) {
                            step("throw", err);
                        });
                    }
                }

                return step("next");
            });
        };
    }

    var ENDPOINT_URL = '/stash/rest/ui/latest/accessibility/color-filter';

    jest.mock('bitbucket/internal/util/client-storage');
    jest.mock('bitbucket/internal/util/ajax');
    jest.mock('bitbucket/internal/util/analytics');

    describe('getColorFilterURL', function () {
        it('gets the correct API URL', function () {
            var url = (0, _colorFilterMigration._getColorFilterURL)();

            expect(url).toBe(ENDPOINT_URL);
        });
    });

    describe('buildUserStorageKey', function () {
        it('returns null for invalid users', function () {
            var userStorageKey = (0, _colorFilterMigration._buildUserStorageKey)();

            expect(userStorageKey).toBeNull();

            userStorageKey = (0, _colorFilterMigration._buildUserStorageKey)({});

            expect(userStorageKey).toBeNull();

            userStorageKey = (0, _colorFilterMigration._buildUserStorageKey)({ currentUser: null });

            expect(userStorageKey).toBeNull();
        });

        it('returns valid storage key', function () {
            var userStorageKey = (0, _colorFilterMigration._buildUserStorageKey)({
                currentUser: {
                    slug: 'testuser'
                }
            });

            expect(userStorageKey).toBe('user-testuser-diff_options');
        });
    });

    describe('isAlternativeColorDiffEnabled', function () {
        beforeEach(function () {
            _clientStorage.getItem.mockClear();
            _clientStorage.setItem.mockClear();
        });

        it('returns false when key is not provided', function () {
            var isEnabled = (0, _colorFilterMigration._isAlternativeColorDiffEnabled)();

            expect(isEnabled).toBeFalsy();
        });

        it('returns correct value when property exists', function () {
            var storageKey = 'fake-storage-key';

            _clientStorage.getItem.mockReturnValue({
                alternateDiffColors: true
            });

            var isEnabled = (0, _colorFilterMigration._isAlternativeColorDiffEnabled)(storageKey);

            expect(_clientStorage.getItem).toHaveBeenCalledWith(storageKey);
            expect(isEnabled).toBeTruthy();

            _clientStorage.getItem.mockReturnValue({
                alternateDiffColors: false
            });

            isEnabled = (0, _colorFilterMigration._isAlternativeColorDiffEnabled)(storageKey);

            expect(_clientStorage.getItem).toHaveBeenCalledWith(storageKey);
            expect(isEnabled).toBeFalsy();
        });
    });

    describe('deleteAlternativeColorDiff', function () {
        beforeEach(function () {
            _clientStorage.getItem.mockClear();
            _clientStorage.setItem.mockClear();
        });

        it('do nothing when key is not provided', function () {
            (0, _colorFilterMigration._deleteAlternativeColorDiff)();

            expect(_clientStorage.getItem).not.toHaveBeenCalled();
            expect(_clientStorage.setItem).not.toHaveBeenCalled();
        });

        it('deletes property when alternative colors exists', function () {
            var storageKey = 'fake-storage-key';
            var data = {
                alternateDiffColors: true
            };

            _clientStorage.getItem.mockReturnValue(data);

            expect(data.alternateDiffColors).toBeTruthy();

            (0, _colorFilterMigration._deleteAlternativeColorDiff)(storageKey);

            expect(_clientStorage.getItem).toHaveBeenCalledWith(storageKey);

            _clientStorage.getItem.mockReturnValue({
                alternateDiffColors: false
            });

            expect(data.alternateDiffColors).toBeUndefined();
        });
    });

    describe('isColorFilterNeverEnabled', function () {
        beforeEach(function () {
            _ajax.rest.mockClear();
            _ajax.rest.mockReturnValue({
                done: jest.fn(),
                fail: jest.fn()
            });
        });

        it('call endpoint with correct URL', _asyncToGenerator( /*#__PURE__*/regeneratorRuntime.mark(function _callee() {
            var colorFilterPromise;
            return regeneratorRuntime.wrap(function _callee$(_context) {
                while (1) {
                    switch (_context.prev = _context.next) {
                        case 0:
                            expect(_ajax.rest).not.toHaveBeenCalled();

                            colorFilterPromise = (0, _colorFilterMigration._isColorFilterNeverEnabled)();


                            expect(_ajax.rest).toHaveBeenCalledWith(expect.objectContaining({
                                url: ENDPOINT_URL
                            }));

                            // make sure the promise resolves on 200
                            _ajax.rest.mock.calls[0][0].statusCode['200']();

                            _context.t0 = expect;
                            _context.next = 7;
                            return colorFilterPromise;

                        case 7:
                            _context.t1 = _context.sent;
                            (0, _context.t0)(_context.t1).toBeFalsy();

                        case 9:
                        case 'end':
                            return _context.stop();
                    }
                }
            }, _callee, undefined);
        })));

        it('rejects the promise when endpoint fails', _asyncToGenerator( /*#__PURE__*/regeneratorRuntime.mark(function _callee2() {
            var colorFilterPromise;
            return regeneratorRuntime.wrap(function _callee2$(_context2) {
                while (1) {
                    switch (_context2.prev = _context2.next) {
                        case 0:
                            expect(_ajax.rest).not.toHaveBeenCalled();

                            colorFilterPromise = (0, _colorFilterMigration._isColorFilterNeverEnabled)();


                            expect(_ajax.rest).toHaveBeenCalledWith(expect.objectContaining({
                                url: ENDPOINT_URL
                            }));

                            // make sure the promise rejects on any other case
                            _ajax.rest.mock.calls[0][0].statusCode['*']();

                            _context2.prev = 4;
                            _context2.next = 7;
                            return colorFilterPromise;

                        case 7:
                            _context2.next = 12;
                            break;

                        case 9:
                            _context2.prev = 9;
                            _context2.t0 = _context2['catch'](4);

                            expect(_context2.t0).toEqual('Unable to verify user settings for migration');

                        case 12:
                        case 'end':
                            return _context2.stop();
                    }
                }
            }, _callee2, undefined, [[4, 9]]);
        })));
    });

    describe('setDefaultColorFilter', function () {
        beforeEach(function () {
            _ajax.rest.mockClear();
            _ajax.rest.mockReturnValue({
                done: jest.fn(),
                fail: jest.fn()
            });
            _analytics.add.mockClear();
        });

        it('call endpoint with correct URL', _asyncToGenerator( /*#__PURE__*/regeneratorRuntime.mark(function _callee3() {
            var colorFilterPromise;
            return regeneratorRuntime.wrap(function _callee3$(_context3) {
                while (1) {
                    switch (_context3.prev = _context3.next) {
                        case 0:
                            expect(_ajax.rest).not.toHaveBeenCalled();

                            colorFilterPromise = (0, _colorFilterMigration._setDefaultColorFilter)();


                            expect(_ajax.rest).toHaveBeenCalledWith(expect.objectContaining({
                                url: ENDPOINT_URL,
                                type: 'PUT',
                                data: {
                                    colorFilter: 'red-green'
                                }
                            }));

                            // make sure the promise resolves on 200
                            _ajax.rest.mock.calls[0][0].statusCode['200']();

                            _context3.t0 = expect;
                            _context3.next = 7;
                            return colorFilterPromise;

                        case 7:
                            _context3.t1 = _context3.sent;
                            (0, _context3.t0)(_context3.t1).toBeFalsy();

                        case 9:
                        case 'end':
                            return _context3.stop();
                    }
                }
            }, _callee3, undefined);
        })));

        it('calls analytics when successful', _asyncToGenerator( /*#__PURE__*/regeneratorRuntime.mark(function _callee4() {
            var fakeUserId, colorFilterPromise;
            return regeneratorRuntime.wrap(function _callee4$(_context4) {
                while (1) {
                    switch (_context4.prev = _context4.next) {
                        case 0:
                            fakeUserId = 'user123';


                            expect(_ajax.rest).not.toHaveBeenCalled();

                            colorFilterPromise = (0, _colorFilterMigration._setDefaultColorFilter)(fakeUserId);


                            expect(_ajax.rest).toHaveBeenCalledWith(expect.objectContaining({
                                url: ENDPOINT_URL,
                                type: 'PUT',
                                data: {
                                    colorFilter: 'red-green'
                                }
                            }));

                            // make sure the promise resolves on 200
                            _ajax.rest.mock.calls[0][0].statusCode['200']();

                            _context4.t0 = expect;
                            _context4.next = 8;
                            return colorFilterPromise;

                        case 8:
                            _context4.t1 = _context4.sent;
                            (0, _context4.t0)(_context4.t1).toBeFalsy();


                            expect(_analytics.add).toHaveBeenCalledWith('stash.client.color.filter.user.migrated', {
                                'user.id': fakeUserId
                            });

                        case 11:
                        case 'end':
                            return _context4.stop();
                    }
                }
            }, _callee4, undefined);
        })));

        it('rejects the promise when endpoint fails', _asyncToGenerator( /*#__PURE__*/regeneratorRuntime.mark(function _callee5() {
            var colorFilterPromise;
            return regeneratorRuntime.wrap(function _callee5$(_context5) {
                while (1) {
                    switch (_context5.prev = _context5.next) {
                        case 0:
                            expect(_ajax.rest).not.toHaveBeenCalled();

                            colorFilterPromise = (0, _colorFilterMigration._setDefaultColorFilter)();


                            expect(_ajax.rest).toHaveBeenCalledWith(expect.objectContaining({
                                url: ENDPOINT_URL,
                                type: 'PUT',
                                data: {
                                    colorFilter: 'red-green'
                                }
                            }));

                            // make sure the promise rejects on error
                            _ajax.rest.mock.calls[0][0].statusCode['*']();

                            _context5.prev = 4;
                            _context5.next = 7;
                            return colorFilterPromise;

                        case 7:
                            _context5.next = 12;
                            break;

                        case 9:
                            _context5.prev = 9;
                            _context5.t0 = _context5['catch'](4);

                            expect(_context5.t0).toEqual('Unable to set user default setting');

                        case 12:
                        case 'end':
                            return _context5.stop();
                    }
                }
            }, _callee5, undefined, [[4, 9]]);
        })));
    });
});