define('bitbucket/internal/feature/repository/revision-reference-selector/revision-reference-selector', ['module', 'exports', '@atlassian/aui', 'jquery', 'lodash', 'bitbucket/internal/model/page-state', 'bitbucket/internal/model/repository', 'bitbucket/internal/model/revision-reference', 'bitbucket/internal/util/ajax', 'bitbucket/internal/util/events', 'bitbucket/internal/widget/searchable-selector/searchable-selector', 'bitbucket/util/navbuilder'], function (module, exports, _aui, _jquery, _lodash, _pageState, _repository, _revisionReference, _ajax, _events, _searchableSelector, _navbuilder) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });

    var AJS = _interopRequireWildcard(_aui);

    var _jquery2 = _interopRequireDefault(_jquery);

    var _lodash2 = _interopRequireDefault(_lodash);

    var _pageState2 = _interopRequireDefault(_pageState);

    var _repository2 = _interopRequireDefault(_repository);

    var _revisionReference2 = _interopRequireDefault(_revisionReference);

    var ajax = _interopRequireWildcard(_ajax);

    var _events2 = _interopRequireDefault(_events);

    var _searchableSelector2 = _interopRequireDefault(_searchableSelector);

    var nav = _interopRequireWildcard(_navbuilder);

    function _interopRequireDefault(obj) {
        return obj && obj.__esModule ? obj : {
            default: obj
        };
    }

    function _interopRequireWildcard(obj) {
        if (obj && obj.__esModule) {
            return obj;
        } else {
            var newObj = {};

            if (obj != null) {
                for (var key in obj) {
                    if (Object.prototype.hasOwnProperty.call(obj, key)) newObj[key] = obj[key];
                }
            }

            newObj.default = obj;
            return newObj;
        }
    }

    /**
     * A searchable selector for choosing RevisionReferences (branches, tags & commits)
     * @extends {SearchableSelector}
     * @return {RevisionReferenceSelector}  The new RevisionReferenceSelector instance
     *
     * @param {HTMLElement|jQuery}  trigger     The trigger (usually a button) to bind opening the selector to.
     * @param {Object}              options     A hash of options, valid options are specified in RevisionReferenceSelector.prototype.defaults
     */
    function RevisionReferenceSelector(trigger, options) {
        return this.init.apply(this, arguments);
    }

    _jquery2.default.extend(RevisionReferenceSelector.prototype, _searchableSelector2.default.prototype);

    /**
     * Returns an object containing the query parameters to retrieve
     * a commit, or null to cancel the request.
     * @return  {object}    The query parameters key/values.
     */
    RevisionReferenceSelector.prototype.commitQueryParamsBuilder = function (searchTerm) {
        // do not fetch commits if there is no search term
        if (!searchTerm) {
            return null;
        }

        return {};
    };

    /**
     * Default options.
     * All options can also be specified as functions that return the desired type (except params that expect a function).
     * Full option documentation can be found on SearchableSelector.prototype.defaults
     * @inheritDocs
     *
     * @param {Repository}  repository      The repository that the selector will retrieve revisions from
     * @param {Object}      show            A hash of which tabs to show or hide
     */
    RevisionReferenceSelector.prototype.defaults = _jquery2.default.extend(true, {}, _searchableSelector2.default.prototype.defaults, {
        tabs: [{
            label: 'Branches',
            type: _revisionReference2.default.type.BRANCH,
            resultsTemplate: bitbucket.internal.feature.repository.revisionReferenceSelector.revisionReferenceSelector.revisionReferenceSelectorBranchResults,
            searchPlaceholder: AJS.I18n.getText('bitbucket.web.revisionref.selector.branch.search.placeholder')
        }, {
            label: 'Tags',
            type: _revisionReference2.default.type.TAG,
            resultsTemplate: bitbucket.internal.feature.repository.revisionReferenceSelector.revisionReferenceSelector.revisionReferenceSelectorTagResults,
            searchPlaceholder: AJS.I18n.getText('bitbucket.web.revisionref.selector.tag.search.placeholder')
        }, {
            label: 'Commits',
            type: _revisionReference2.default.type.COMMIT,
            queryParamsBuilder: RevisionReferenceSelector.prototype.commitQueryParamsBuilder,
            resultsTemplate: bitbucket.internal.feature.repository.revisionReferenceSelector.revisionReferenceSelector.revisionReferenceSelectorCommitResults,
            searchPlaceholder: AJS.I18n.getText('bitbucket.web.revisionref.selector.commit.search.placeholder')
        }],
        queryParamKey: 'filterText',
        namespace: 'revision-reference-selector',
        itemSelectedEvent: 'bitbucket.internal.DO_NOT_USE.feature.repository.revisionReferenceSelector.revisionRefChanged',
        itemUnselectedEvent: 'bitbucket.internal.DO_NOT_USE.feature.repository.revisionReferenceSelector.revisionRefUnselected',
        itemDataKey: 'revision-ref',
        triggerContentTemplate: bitbucket.internal.feature.repository.revisionReferenceSelector.revisionReferenceSelector.revisionReferenceSelectorTriggerContent,
        extraClasses: 'revision-reference-selector',
        repository: function repository() {
            return _pageState2.default.getRepository();
        },
        show: { branches: true, tags: true, commits: false },
        dataTransform: RevisionReferenceSelector.prototype._transformResults,
        postOptionsInit: function postOptionsInit() {
            this.setRepository(this._getOptionVal('repository'));
        },
        paginationContext: 'revision-reference-selector'
    });

    /**
     * Initialise the selector
     * @override
     * @return {RevisionReferenceSelector} The initialised RevisionReferenceSelector.
     *
     * @param {HTMLElement|jQuery}  trigger     The trigger (usually a button) to bind opening the selector to.
     * @param {Object}              options     A hash of options, valid options are specified in RevisionReferenceSelector.prototype.defaults
     */
    RevisionReferenceSelector.prototype.init = function (trigger, options) {
        _searchableSelector2.default.prototype.init.apply(this, arguments);

        return this;
    };

    /**
     * Merge the supplied options with the defaults and remove tabs that aren't going to be shown.
     * @override
     *
     * @param {Object}  options A hash of options, valid options are specified in RevisionReferenceSelector.prototype.defaults
     */
    RevisionReferenceSelector.prototype.setOptions = function (options) {
        if (options.extraClasses) {
            options.extraClasses = this.defaults.extraClasses + ' ' + _jquery2.default.trim(options.extraClasses);
        }
        options = _jquery2.default.extend(true, {}, this.defaults, options);
        var typesMap = {
            branches: _revisionReference2.default.type.BRANCH.id,
            tags: _revisionReference2.default.type.TAG.id,
            commits: _revisionReference2.default.type.COMMIT.id
        };
        var typesToShow = _lodash2.default.filter(typesMap, function (type, key) {
            //Only show types with enabled in the `show` options.
            return options.show[key];
        });

        //Remove any tabs whose type is not in `typesToShow`
        options.tabs = _lodash2.default.filter(options.tabs, function (tab) {
            return _lodash2.default.includes(typesToShow, tab.type.id);
        });

        this.options = options;
    };

    /**
     * Build a RevisionReference from the metadata on the trigger.
     * @override
     * @return {RevisionReference} The newly created RevisionReference
     */
    RevisionReferenceSelector.prototype._getItemFromTrigger = function () {
        var $triggerItem = this.$trigger.find('.name');

        return new _revisionReference2.default(_jquery2.default.extend({}, this._buildObjectFromElementDataAttributes($triggerItem), {
            displayId: $triggerItem.text(),
            repository: this.repository
        }));
    };

    /**
     * Handles the Ajax request and data transform
     * @override
     * @return {Promise}
     *
     * @param {object} queryParams  The search query parameters.
     */
    RevisionReferenceSelector.prototype.handleRequest = function (queryParams) {
        var _this = this;

        var url = void 0;

        switch (this._getCurrentType().id) {
            case _revisionReference2.default.type.BRANCH.id:
                url = this.getBranchesUrl();
                break;
            case _revisionReference2.default.type.TAG.id:
                url = this.getTagsUrl();
                break;
            case _revisionReference2.default.type.COMMIT.id:
                url = this.getCommitUrl(this._getSearchTerm());
                break;
        }

        var noResults = function noResults() {
            return _jquery2.default.Deferred().resolve({
                isLastPage: true,
                size: 0,
                values: []
            });
        };

        // URL is null, do not send request and show no results
        if (!url) {
            return noResults();
        }

        return ajax.rest({
            url: url,
            data: queryParams,
            statusCode: ajax.ignore404WithinRepository()
        }).then(function (data) {
            return _this._transformResults(data);
        },
        // show no results in case of handled ajax failure
        function () {
            return noResults();
        });
    };

    /**
     * Transform the result of the REST API to match the expected data shape.
     * @return {Object} The modified collection of results
     *
     * @param {Object} rawData The raw data from the REST endpoint
     */
    RevisionReferenceSelector.prototype._transformResults = function (rawData) {
        var data = rawData;

        if (this._getCurrentType().id === _revisionReference2.default.type.COMMIT.id) {
            // change the shape of the response for commit as there
            // is only ever one or zero in results
            data = {
                isLastPage: true,
                size: 1,
                start: 0,
                values: [rawData]
            };
        }

        return this._addRefTypeAndRepositoryToResults(data);
    };

    /**
     * Add the current revision reference type and the repository to each item in a collection of results.
     * Is used as the dataTransform function on REST and preloaded results.
     * @return {Object} The modified collection of results
     *
     * @param {Object} results The collection of results
     */
    RevisionReferenceSelector.prototype._addRefTypeAndRepositoryToResults = function (results) {
        if (results && results.values) {
            var newResults = _jquery2.default.extend(true, {}, results); //Deep clone;
            var refType = this._getCurrentType();

            _lodash2.default.forEach(newResults.values, _lodash2.default.bind(function (ref) {
                ref.type = refType;
                if (!ref.repository) {
                    ref.repository = this.repository && this.repository.toJSON();
                }
            }, this));

            return newResults;
        }

        return results;
    };

    /**
     * Get the url for the branches REST endpoint for the current repository
     * @return  {string}    The url to the rest endpoint for branches
     */
    RevisionReferenceSelector.prototype.getBranchesUrl = function () {
        return nav.rest().project(this.repository.getProject()).repo(this.repository).branches().build();
    };

    /**
     * Get the url for the tags REST endpoint for the current repository
     * @return  {string}    The url to the rest endpoint for tags
     */
    RevisionReferenceSelector.prototype.getTagsUrl = function () {
        return nav.rest().project(this.repository.getProject()).repo(this.repository).tags().build();
    };

    /**
     * Get the url for the commit REST endpoint for the current repository
     * for a specific commit ID
     * @param  {string}  searchText  The search term corresponding to the commit ID
     * @return  {string}    The url to the rest endpoint for commits
     */
    RevisionReferenceSelector.prototype.getCommitUrl = function (searchText) {
        // If the search text does not look like a commit ID
        // return a null URL
        if (!searchText.match(/^[a-fA-F0-9]+$/)) {
            return null;
        }

        return nav.rest().project(this.repository.getProject()).repo(this.repository).commit(searchText).build();
    };

    /**
     * Get the current repository
     * @return {Repository}     The current repository
     */
    RevisionReferenceSelector.prototype.getRepository = function () {
        return this.repository;
    };

    /**
     * Update the current repository.
     * Resets state for the current scrollable and trigger and hides the dialog.
     *
     * @param {Repository}  repository  The new repository
     */
    RevisionReferenceSelector.prototype.setRepository = function (repository) {
        var currentRepository = this.repository;

        if (repository instanceof _repository2.default && !repository.isEqual(currentRepository)) {
            //Changing repository to the same repository should be a no-op.
            this.repository = repository;

            if (currentRepository) {
                //Only reset the scrollable and trigger, close the dialog and trigger the event when we are changing between repositories, not setting the repo for the first time.
                var currentScrollable = this._getCurrentScrollable();

                if (currentScrollable) {
                    //We don't call _populateScrollable, because after changing repository it doesn't make sense to show the preload data
                    this._emptyScrollable(currentScrollable);
                    currentScrollable.init();
                }
                this.clearSelection();
                this.dialog.open = false;

                _events2.default.trigger('bitbucket.internal.DO_NOT_USE.feature.repository.revisionReferenceSelector.repoChanged', this, repository, this._getOptionVal('context'));
            }
        }
    };

    /**
     * Get the RevisionReference type of the current tab.
     * @return {Object} The current tab type
     */
    RevisionReferenceSelector.prototype._getCurrentType = function () {
        return this.tabs[this.currentTabId || 0].type;
    };

    /**
     * Set the selected item.
     * Updates the trigger and fires the event if the item is different to the previous item.
     * @override
     *
     * @param {Object} revisionRef The RevisionReference to select.
     */
    RevisionReferenceSelector.prototype.setSelectedItem = function (revisionRef) {
        if (revisionRef instanceof _revisionReference2.default && !revisionRef.isEqual(this._selectedItem)) {
            this._itemSelected(revisionRef);
        }
    };

    RevisionReferenceSelector.prototype.clearSelection = function () {
        _searchableSelector2.default.prototype.clearSelection.apply(this, arguments);
        // null arg in place of revisionRef from the itemSelectedEvent
        _events2.default.trigger(this._getOptionVal('itemUnselectedEvent'), this, null, this._getOptionVal('context'));
    };

    /**
     * Handle an item being selected.
     * This creates a new RevisionReference from the item data,
     * triggers the 'stash.feature.repository.revisionReferenceSelector.revisionRefChanged' event with the new RevisionReference,
     * sets the selectedItem to the new RevisionReference and updates the trigger and form field (if supplied)
     * @override
     *
     * @param {Object|RevisionReference}  refDataOrRevisionReference     The JSON data or a RevisionReference for the selected item.
     */
    RevisionReferenceSelector.prototype._itemSelected = function (refDataOrRevisionReference) {
        var refData;
        var ref;

        if (refDataOrRevisionReference instanceof _revisionReference2.default) {
            refData = refDataOrRevisionReference.toJSON();
            ref = refDataOrRevisionReference;
        } else {
            refData = _lodash2.default.pick(refDataOrRevisionReference, _lodash2.default.keys(_revisionReference2.default.prototype.namedAttributes));
            ref = new _revisionReference2.default(refData);
        }

        this._selectedItem = ref;

        if (this._getOptionVal('field')) {
            (0, _jquery2.default)(this._getOptionVal('field')).val(refData.id).trigger('change');
        }

        var titleContent = bitbucket.internal.feature.repository.revisionReferenceSelector.revisionReferenceSelector.revisionReferenceSelectorTitle({
            ref: refData
        });

        this.updateTrigger({ ref: refData }, titleContent);

        _events2.default.trigger(this._getOptionVal('itemSelectedEvent'), this, ref, this._getOptionVal('context'));
    };

    exports.default = RevisionReferenceSelector;
    module.exports = exports['default'];
});