define('bitbucket/internal/feature/repository/source-target-selector/source-target-selector', ['module', 'exports', '@atlassian/aui', 'jquery', 'lodash', 'bitbucket/internal/feature/repository/branch-selector/branch-selector', 'bitbucket/internal/feature/repository/related-repository-selector/related-repository-selector', 'bitbucket/internal/util/ajax', 'bitbucket/internal/util/dom-event', 'bitbucket/internal/util/events', 'bitbucket/util/navbuilder'], function (module, exports, _aui, _jquery, _lodash, _branchSelector, _relatedRepositorySelector, _ajax, _domEvent, _events, _navbuilder) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });

    var _jquery2 = _interopRequireDefault(_jquery);

    var _lodash2 = _interopRequireDefault(_lodash);

    var _branchSelector2 = _interopRequireDefault(_branchSelector);

    var _relatedRepositorySelector2 = _interopRequireDefault(_relatedRepositorySelector);

    var ajax = _interopRequireWildcard(_ajax);

    var _domEvent2 = _interopRequireDefault(_domEvent);

    var _events2 = _interopRequireDefault(_events);

    var nav = _interopRequireWildcard(_navbuilder);

    function _interopRequireWildcard(obj) {
        if (obj && obj.__esModule) {
            return obj;
        } else {
            var newObj = {};

            if (obj != null) {
                for (var key in obj) {
                    if (Object.prototype.hasOwnProperty.call(obj, key)) newObj[key] = obj[key];
                }
            }

            newObj.default = obj;
            return newObj;
        }
    }

    function _interopRequireDefault(obj) {
        return obj && obj.__esModule ? obj : {
            default: obj
        };
    }

    /**
     * A selector for picking source, target branches/repositories
     *
     * @param $container
     * @param sourceRepository
     * @param targetRepository
     * @param additionalPreloadRepositories
     * @param options
     * @constructor
     */
    function SourceTargetSelector($container, sourceRepository, targetRepository, additionalPreloadRepositories, options) {
        this.init.apply(this, arguments);
    }

    SourceTargetSelector.prototype.defaults = {
        showCommitBadges: true,
        source: {
            showTags: false,
            showCommits: false
        },
        target: {
            showTags: false,
            showCommits: false
        }
    };

    SourceTargetSelector.prototype.init = function ($container, sourceRepository, targetRepository, additionalPreloadRepositories, options) {
        var self = this;

        self.refSelectors = {};
        self.options = _lodash2.default.defaultsDeep(options, self.defaults);

        var contexts = [{ name: 'source', repository: sourceRepository }, { name: 'target', repository: targetRepository }];

        var preloadedRepoPage = _relatedRepositorySelector2.default.constructDataPageFromPreloadArray(_lodash2.default.chain(contexts).map('repository').union(additionalPreloadRepositories).compact().uniqBy(function (repo) {
            return repo.getId();
        }) // remove dupes
        .invokeMap('toJSON').value());

        _lodash2.default.forEach(contexts, function (context) {
            var $branchSelectorTrigger = (0, _jquery2.default)('.' + context.name + 'Branch', $container);
            var $branchInput = (0, _jquery2.default)('#' + context.name + 'Branch-field', $container);

            var $repoSelectorTrigger = (0, _jquery2.default)('.' + context.name + 'Repo', $container);
            var $repoInput = (0, _jquery2.default)('#' + context.name + 'Repo-field', $container);

            // I would love to use template strings, but for some reason, I18n
            // doesn't want to work with them here :(
            var repoSelectorAriaLabel = context.name == 'source' ? _aui.I18n.getText('bitbucket.web.revisionref.selector.source.repository.label') : _aui.I18n.getText('bitbucket.web.revisionref.selector.destination.repository.label');

            var branchSelectorAriaLabel = context.name == 'source' ? _aui.I18n.getText('bitbucket.web.revisionref.selector.source.branch.label') : _aui.I18n.getText('bitbucket.web.revisionref.selector.destination.branch.label');

            if (self.options[context.name].showTags && !self.options[context.name].showCommits) {
                branchSelectorAriaLabel = context.name == 'source' ? _aui.I18n.getText('bitbucket.web.revisionref.selector.source.branch.or.tag.label') : _aui.I18n.getText('bitbucket.web.revisionref.selector.destination.branch.or.tag.label');
            } else if (!self.options[context.name].showTags && self.options[context.name].showCommits) {
                branchSelectorAriaLabel = context.name == 'source' ? _aui.I18n.getText('bitbucket.web.revisionref.selector.source.branch.or.commit.label') : _aui.I18n.getText('bitbucket.web.revisionref.selector.destination.branch.or.commit.label');
            } else if (self.options[context.name].showTags && self.options[context.name].showCommits) {
                branchSelectorAriaLabel = context.name == 'source' ? _aui.I18n.getText('bitbucket.web.revisionref.selector.source.branch.or.tag.or.commit.label') : _aui.I18n.getText('bitbucket.web.revisionref.selector.destination.branch.or.tag.or.commit.label');
            }

            var refSelector = {
                // n.b. hide the spinner to avoid display conflicts with the plugin point after the branch input
                $headCommitSpinner: (0, _jquery2.default)("<div class='spinner-wrapper'/>").hide().insertAfter($branchInput),
                branchSelector: new _branchSelector2.default($branchSelectorTrigger, {
                    id: context.name + 'BranchDialog',
                    ariaLabel: branchSelectorAriaLabel,
                    context: context.name,
                    repository: context.repository,
                    field: $branchInput,
                    show: {
                        branches: true,
                        tags: self.options[context.name].showTags,
                        commits: self.options[context.name].showCommits
                    },
                    triggerPlaceholder: self.options[context.name].triggerPlaceholder,
                    paginationContext: 'source-target-selector'
                }),
                repoSelector: new _relatedRepositorySelector2.default($repoSelectorTrigger, {
                    id: context.name + 'RepoDialog',
                    ariaLabel: repoSelectorAriaLabel,
                    context: context.name,
                    repository: context.repository,
                    field: $repoInput,
                    preloadData: preloadedRepoPage
                }),
                getBranch: function getBranch() {
                    return this.branchSelector.getSelectedItem();
                },
                getRepo: function getRepo() {
                    return this.repoSelector.getSelectedItem();
                },
                getSelection: function getSelection() {
                    return {
                        repository: this.getRepo(),
                        branch: this.getBranch()
                    };
                },
                setSelection: function setSelection(selection) {
                    // Repo selection has to be first
                    if (selection.repository) {
                        this.repoSelector.setSelectedItem(selection.repository);
                    } else {
                        this.repoSelector.clearSelection();
                    }

                    if (selection.branch) {
                        this.branchSelector.setSelectedItem(selection.branch);
                    } else {
                        this.branchSelector.clearSelection();
                        this._getCommitBadge().empty();
                    }
                },
                _getCommitBadge: function _getCommitBadge() {
                    return this.branchSelector.$trigger.siblings('.commit-badge-detailed');
                }
            };

            var $commitHash = refSelector._getCommitBadge().find('.commitid');
            $commitHash.tipsy({
                opacity: 0.98,
                title: 'data-commit-message',
                className: 'commit-preview-tipsy'
            });

            self.refSelectors[context.name] = refSelector;
        });

        $container.find('.swap-button').on('click', _domEvent2.default.preventDefault(this.swap.bind(this)));

        _events2.default.on('bitbucket.internal.DO_NOT_USE.feature.repository.revisionReferenceSelector.revisionRefChanged', function (revisionRef, context) {
            var refSelector = self.refSelectors[context];

            if (self.options.showCommitBadges) {
                self._updateCommitBadge(refSelector, revisionRef);
            }

            _events2.default.trigger('bitbucket.internal.DO_NOT_USE.feature.repository.sourceTargetSelector.' + context + '.revisionRefChanged', self, revisionRef);
        });

        _events2.default.on('bitbucket.internal.DO_NOT_USE.feature.repository.repositorySelector.repositoryChanged', function (repository, context) {
            var refSelector = self.refSelectors[context];
            refSelector.branchSelector.setRepository(repository);

            if (self.options.showCommitBadges) {
                self._updateCommitBadge(refSelector, null);
            }

            _events2.default.trigger('bitbucket.internal.DO_NOT_USE.feature.repository.sourceTargetSelector.' + context + '.repositoryChanged', self, repository);
        });

        _events2.default.on('bitbucket.internal.DO_NOT_USE.feature.repository.revisionReferenceSelector.revisionRefUnselected', function (revisionRef, context) {
            var refSelector = self.refSelectors[context];

            if (self.options.showCommitBadges) {
                self._updateCommitBadge(refSelector, revisionRef);
            }

            _events2.default.trigger('bitbucket.internal.DO_NOT_USE.feature.repository.sourceTargetSelector.' + context + '.revisionRefUnselected', self, revisionRef);
        });

        return self;
    };

    SourceTargetSelector.prototype.swap = function () {
        var sourceSelector = this.refSelectors.source;
        var targetSelector = this.refSelectors.target;
        var sourceSelection = sourceSelector.getSelection();

        sourceSelector.setSelection(targetSelector.getSelection());
        targetSelector.setSelection(sourceSelection);
    };

    SourceTargetSelector.prototype._updateCommitBadge = function (refSelector, revisionRef) {
        var self = this;
        var $commitBadge = refSelector._getCommitBadge();

        $commitBadge.addClass('fade').empty();

        if (revisionRef) {
            refSelector.$headCommitSpinner.show().spin('small');
            var repo = refSelector.getRepo();
            ajax.rest({
                url: nav.rest().project(repo.getProject()).repo(repo).commit(revisionRef.getComparisonCommit()).build()
            }).done(function (commit) {
                var $newCommitBadge = (0, _jquery2.default)(bitbucket.internal.feature.commit.commitBadge.commitBadge.oneline({
                    commit: commit,
                    linkAuthor: false,
                    messageTooltip: true,
                    repository: repo.toJSON()
                }));

                $commitBadge.append($newCommitBadge);
                $commitBadge.removeClass('fade');
                $commitBadge.find('.commitid').tipsy({
                    opacity: 0.98,
                    title: 'data-commit-message',
                    className: 'commit-preview-tipsy'
                });
            }).always(function () {
                refSelector.$headCommitSpinner.spinStop().hide();
            });
        } else {
            $commitBadge.empty();
        }
    };

    SourceTargetSelector.prototype.branchesSelected = function () {
        return !!(this.refSelectors.source.getBranch() && this.refSelectors.target.getBranch());
    };

    SourceTargetSelector.prototype.refsAreEqual = function () {
        var sourceRef = this.refSelectors.source.getBranch();
        var targetRef = this.refSelectors.target.getBranch();

        return !!(sourceRef && sourceRef.isEqual(targetRef));
    };

    SourceTargetSelector.prototype.getSourceRepository = function () {
        return this.refSelectors.source.getRepo();
    };

    SourceTargetSelector.prototype.getTargetRepository = function () {
        return this.refSelectors.target.getRepo();
    };

    SourceTargetSelector.prototype.getSourceBranch = function () {
        return this.refSelectors.source.getBranch();
    };

    SourceTargetSelector.prototype.getTargetBranch = function () {
        return this.refSelectors.target.getBranch();
    };

    exports.default = SourceTargetSelector;
    module.exports = exports['default'];
});