define('bitbucket/internal/feature/websudo-banner/websudo-banner', ['exports', 'jquery', 'moment', 'bitbucket/util/navbuilder', 'bitbucket/util/server'], function (exports, _jquery, _moment, _navbuilder, _server) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports._bindExpiredReauthenticate = exports._bindReauthenticate = exports._bindDropAccess = exports._checkExpiryAndUpdateContent = exports.onReady = undefined;

    var _jquery2 = _interopRequireDefault(_jquery);

    var _moment2 = _interopRequireDefault(_moment);

    var nav = _interopRequireWildcard(_navbuilder);

    function _interopRequireWildcard(obj) {
        if (obj && obj.__esModule) {
            return obj;
        } else {
            var newObj = {};

            if (obj != null) {
                for (var key in obj) {
                    if (Object.prototype.hasOwnProperty.call(obj, key)) newObj[key] = obj[key];
                }
            }

            newObj.default = obj;
            return newObj;
        }
    }

    function _interopRequireDefault(obj) {
        return obj && obj.__esModule ? obj : {
            default: obj
        };
    }

    var ABOUT_TO_EXPIRE_MINS = 2;
    var SESSION_POLL_INTERVAL = 5000;
    var EXPIRY_FORMAT = 'hh:mm A';
    var urlBuilder = nav.rest('ui').addPathComponents('websudo-session');
    // used to update banner content when session is about to expire
    var aboutToExpireTimer = null;
    // used to update banner content when session is expired
    var expiryTimer = null;
    // used to poll session if its valid
    var reAuthPollTimer = null;

    /**
     * function to bind the click event on drop access button
     * on click, sends the DELETE request
     * @param {*} $container jQuery element for the websudo banner
     */
    function bindDropAccess($container) {
        $container.on('click', '.websudo-banner-drop-access', function () {
            return (0, _server.rest)({
                url: urlBuilder.build(),
                method: _server.method.DELETE
            }).done(function () {
                if ($container.hasClass('websudo-required')) {
                    // redirect to the dashboard if this is a websudo protected page
                    window.location.assign(nav.dashboard().build());
                } else {
                    // remove the banner
                    $container.remove();
                }
            });
        });
    }

    /**
     * function to bind the click event on reauthenticate button
     * on click, sends the PUT request and on success updates the banner to original content
     * @param {*} $container jQuery element for the websudo banner
     */
    function bindReauthenticate($container) {
        $container.on('click', '.websudo-banner-reauthenticate', function () {
            return (0, _server.rest)({
                url: urlBuilder.build(),
                method: _server.method.PUT
            }).done(function (response) {
                reset($container, response.expiry);
            });
        });
    }

    /**
     * bind click handler with the reauthenticate link
     * start polling to check if new session has been created in another tab
     *
     * @param {*} $container jQuery element for the websudo banner
     */
    function bindExpiredReauthenticate($container) {
        clearInterval(reAuthPollTimer);

        $container.on('click', '.websudo-banner-expired-reauthenticate', function () {
            // start poll and check if session has been updated
            reAuthPollTimer = setInterval(function () {
                (0, _server.rest)({
                    url: urlBuilder.build(),
                    statusCode: {
                        // if no session is present don't do anything
                        404: false
                    }
                }).done(function (response) {
                    clearInterval(reAuthPollTimer);
                    reset($container, response.expiry);
                });
            }, SESSION_POLL_INTERVAL);
        });
    }

    /**
     * function to update the banner content when its about to expire
     *
     * @param {String} expireTime timestamp
     */
    function renderAboutToExpireContent(expireTime) {
        (0, _jquery2.default)('.websudo-banner-content').html(bitbucket.internal.feature.websudoBanner.content({
            isAboutToExpire: true,
            expireTime: expireTime
        }));
    }

    /**
     * function to render the expired content that includes reauthenticate link
     * @param {*} $container jQuery element for the websudo banner
     */
    function renderExpiredContent($container) {
        (0, _jquery2.default)('.websudo-banner-content').html(bitbucket.internal.feature.websudoBanner.content({
            isExpired: true,
            reauthenticateRef: nav.newBuilder('websudo').build()
        }));
    }

    /**
     * function to check:
     * if about to expire and then update the content with expiry time
     * if expired then on reauthenticate click, open websudo in new tab and poll
     */
    function checkExpiryAndUpdateContent($container) {
        clearTimeout(aboutToExpireTimer);
        clearTimeout(expiryTimer);

        var timestamp = $container.find('#websudo-banner-expiry').data('expiry');
        var currentTimestamp = (0, _moment2.default)();
        var expiryTimestamp = (0, _moment2.default)(timestamp);

        // check how much time is left
        // if less than 2 mins - show diff content
        // else - add a timer to show diff content when 2 mins are left, i.e minutesLeft - 2
        var duration = _moment2.default.duration(expiryTimestamp.diff(currentTimestamp));
        var minutesLeft = duration.asMinutes();

        if (minutesLeft > 0) {
            if (minutesLeft <= ABOUT_TO_EXPIRE_MINS) {
                renderAboutToExpireContent(expiryTimestamp.format(EXPIRY_FORMAT));
            } else {
                // warning 2 mins before the session expires
                aboutToExpireTimer = setTimeout(function () {
                    renderAboutToExpireContent(expiryTimestamp.format(EXPIRY_FORMAT));
                    // minutes in milli seconds
                }, (minutesLeft - ABOUT_TO_EXPIRE_MINS) * 60000);
            }

            // once the session expires render the expired content
            expiryTimer = setTimeout(function () {
                renderExpiredContent($container);
            }, minutesLeft * 60000);
        } else {
            // already expired
            renderExpiredContent($container);
        }
    }

    /**
     * function to update the banner contents
     * @param {*} $container jQuery element for the websudo banner
     * @param {Number} expiry timestamp
     */
    function reset($container, expiry) {
        // update the expiry data attribute
        $container.find('#websudo-banner-expiry').data('expiry', expiry);
        // update content to original
        $container.find('.websudo-banner-content').html(bitbucket.internal.feature.websudoBanner.content({}));
        // check and set timer
        checkExpiryAndUpdateContent($container);
    }

    function onReady() {
        var $container = (0, _jquery2.default)('#websudo-banner');

        if ($container.length) {
            checkExpiryAndUpdateContent($container);

            bindDropAccess($container);
            bindReauthenticate($container);
            bindExpiredReauthenticate($container);
        }
    }

    exports.onReady = onReady;
    exports._checkExpiryAndUpdateContent = checkExpiryAndUpdateContent;
    exports._bindDropAccess = bindDropAccess;
    exports._bindReauthenticate = bindReauthenticate;
    exports._bindExpiredReauthenticate = bindExpiredReauthenticate;
});