define('bitbucket/internal/util/components/react-functional', ['exports', 'react'], function (exports, _react) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.withForwardedRef = exports.getDisplayName = undefined;

    var _react2 = _interopRequireDefault(_react);

    function _interopRequireDefault(obj) {
        return obj && obj.__esModule ? obj : {
            default: obj
        };
    }

    var _extends = Object.assign || function (target) {
        for (var i = 1; i < arguments.length; i++) {
            var source = arguments[i];

            for (var key in source) {
                if (Object.prototype.hasOwnProperty.call(source, key)) {
                    target[key] = source[key];
                }
            }
        }

        return target;
    };

    function _defineProperty(obj, key, value) {
        if (key in obj) {
            Object.defineProperty(obj, key, {
                value: value,
                enumerable: true,
                configurable: true,
                writable: true
            });
        } else {
            obj[key] = value;
        }

        return obj;
    }

    /**
     * Get the display name of a given React component. Falls back to 'Component' when cannot discover the component name.
     *
     * @param Component - A Component used to find a name
     * @returns {string} - Component name
     */
    var getDisplayName = exports.getDisplayName = function getDisplayName(Component) {
        if (typeof Component === 'string') {
            return Component;
        }

        return Component.displayName || Component.name || 'Component';
    };

    /**
     * HoC that creates a Component with a ref forwarding from an inner DOM element. More info about ref forwarding can be
     * found here https://reactjs.org/docs/forwarding-refs.html
     *
     * For example:
     *
     * ```
     * class MyComponent extends Component {
     *     render() {
     *         return (
     *              <div ref={this.props.myForwardRefProp}>
     *                  You can get assess to the inner div outside the MyComponent!
     *              </div>
     *         );
     *     }
     * }
     *
     * const MyComponentWithRef = withForwardedRef('myForwardRefProp')(MyComponent);
     *
     * // Usage
     * class WrapperComponent extends Components {
     *     myRef = React.createRef();
     *
     *     componentDidMount() {
     *         console.log(this.myRef.current) // A div DOM element
     *     }
     *
     *     render() {
     *         <MyComponentWithRef ref={this.myRef} />
     *     }
     * }
     * ```
     *
     * @param {string} forwardedRefProp - A name of the forwarded property that will proxy reference to the Component
     * @returns {function} - Wraped React Component
     */
    var withForwardedRef = exports.withForwardedRef = function withForwardedRef() {
        var forwardedRefProp = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 'forwardedRef';
        return function (Component) {
            var forwardRefComponent = function forwardRefComponent(props, ref) {
                var forwardedRefProps = _defineProperty({}, forwardedRefProp, ref);

                return _react2.default.createElement(Component, _extends({}, props, forwardedRefProps));
            };

            forwardRefComponent.displayName = 'withRef(' + getDisplayName(Component) + ')';

            return (0, _react.forwardRef)(forwardRefComponent);
        };
    };
});