define('bitbucket/internal/util/hooks/use-scroll-to-element-once.test', ['scroll-into-view-if-needed', 'tests/hooks-helper', './use-scroll-to-element-once'], function (_scrollIntoViewIfNeeded, _hooksHelper, _useScrollToElementOnce) {
    'use strict';

    var _scrollIntoViewIfNeeded2 = _interopRequireDefault(_scrollIntoViewIfNeeded);

    function _interopRequireDefault(obj) {
        return obj && obj.__esModule ? obj : {
            default: obj
        };
    }

    jest.mock('scroll-into-view-if-needed');

    describe(_useScrollToElementOnce.useScrollToElementOnce, function () {
        var orgRequestAnimationFrame = window.requestAnimationFrame;
        var requestAnimationFrameCallback = void 0;

        var flushRequestAnimationFrame = function flushRequestAnimationFrame() {
            if (typeof requestAnimationFrameCallback === 'function') {
                requestAnimationFrameCallback();
            }
        };

        beforeAll(function () {
            Object.defineProperty(window, 'requestAnimationFrame', {
                value: function value(callback) {
                    requestAnimationFrameCallback = callback;
                }
            });
        });

        beforeEach(function () {
            _scrollIntoViewIfNeeded2.default.mockReset();
        });

        afterEach(function () {
            requestAnimationFrameCallback = null;
        });

        afterAll(function () {
            Object.defineProperty(window, 'requestAnimationFrame', {
                value: orgRequestAnimationFrame
            });
        });

        it('should call the scrollIntoView function for a given element reference', function () {
            // given
            var myElement = document.createElement('div');
            var myElementRef = {
                current: myElement
            };

            // when
            (0, _hooksHelper.testHook)(function () {
                (0, _useScrollToElementOnce.useScrollToElementOnce)(myElementRef);
            });

            flushRequestAnimationFrame();

            // then
            expect(_scrollIntoViewIfNeeded2.default).toHaveBeenCalledWith(myElement, {
                behavior: 'smooth',
                scrollMode: 'if-needed',
                block: 'nearest',
                inline: 'nearest'
            });
        });

        it('should not call the scrollIntoView function when scrolling is disabled', function () {
            // given
            var myElement = document.createElement('div');
            var myElementRef = {
                current: myElement
            };

            // when
            (0, _hooksHelper.testHook)(function () {
                (0, _useScrollToElementOnce.useScrollToElementOnce)(myElementRef, false);
            });

            // then
            expect(_scrollIntoViewIfNeeded2.default).not.toHaveBeenCalled();
        });

        it('should not call the scrollIntoView function when reference is not set', function () {
            // given
            var myElementRef = {
                current: undefined
            };

            // when
            (0, _hooksHelper.testHook)(function () {
                (0, _useScrollToElementOnce.useScrollToElementOnce)(myElementRef, false);
            });

            flushRequestAnimationFrame();

            // then
            expect(_scrollIntoViewIfNeeded2.default).not.toHaveBeenCalled();
        });

        it('should not call the scrollIntoView function when reference was cleared after we requested animation frame', function () {
            // given
            var myElementRef = {
                current: undefined
            };

            // when
            (0, _hooksHelper.testHook)(function () {
                (0, _useScrollToElementOnce.useScrollToElementOnce)(myElementRef, false);
            });

            flushRequestAnimationFrame();

            // then
            expect(_scrollIntoViewIfNeeded2.default).not.toHaveBeenCalled();
        });

        it('should call the scrollIntoView function with additional params', function () {
            // given
            var myElement = document.createElement('div');
            var myElementRef = {
                current: myElement
            };

            // when
            (0, _hooksHelper.testHook)(function () {
                (0, _useScrollToElementOnce.useScrollToElementOnce)(myElementRef, true, {
                    inline: 'blah',
                    foo: 'bar'
                });
            });

            flushRequestAnimationFrame();

            // then
            expect(_scrollIntoViewIfNeeded2.default).toHaveBeenCalledWith(myElement, {
                behavior: 'smooth',
                scrollMode: 'if-needed',
                block: 'nearest',
                inline: 'blah',
                foo: 'bar'
            });
        });

        it('should call the scrollIntoView function when the modal close event is triggered', function () {
            // given
            jest.useFakeTimers();
            var myElement = document.createElement('div');
            var myElementRef = {
                current: myElement
            };

            // when
            (0, _hooksHelper.testHook)(function () {
                (0, _useScrollToElementOnce.useScrollToElementOnce)(myElementRef);
            });

            flushRequestAnimationFrame();

            // Simulate modal close event
            var event = new Event(_useScrollToElementOnce.SCROLL_TO_ELEMENT_MODAL_CLOSE_EVENT);
            window.dispatchEvent(event);

            // Fast-forward the timer to simulate the delay
            jest.advanceTimersByTime(500);

            flushRequestAnimationFrame();

            // then
            expect(_scrollIntoViewIfNeeded2.default).toHaveBeenCalledTimes(2); // initial call + modal close call

            // Cleanup
            jest.useRealTimers();
        });

        it('should remove the event listener after it executes once', function () {
            // given
            jest.useFakeTimers();
            var myElement = document.createElement('div');
            var myElementRef = {
                current: myElement
            };

            // when
            (0, _hooksHelper.testHook)(function () {
                (0, _useScrollToElementOnce.useScrollToElementOnce)(myElementRef);
            });

            var removeEventListenerSpy = jest.spyOn(window, 'removeEventListener');

            // Simulate modal close event
            var event = new Event(_useScrollToElementOnce.SCROLL_TO_ELEMENT_MODAL_CLOSE_EVENT);
            window.dispatchEvent(event);

            // Fast-forward the timer to simulate the delay
            jest.advanceTimersByTime(500);

            flushRequestAnimationFrame();

            // then
            expect(removeEventListenerSpy).toHaveBeenCalledWith(_useScrollToElementOnce.SCROLL_TO_ELEMENT_MODAL_CLOSE_EVENT, expect.any(Function));

            // Cleanup
            jest.useRealTimers();
        });
    });
});