define('bitbucket/internal/util/rest-actor/rest-actor', ['exports', 'lodash', 'bitbucket/util/server', './requests-registry'], function (exports, _lodash, _server, _requestsRegistry) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.restActorsForTypes = exports.restActorForType = exports.promiseRequestFailure = exports.requestFailure = exports.requestAborted = exports.requestSuccess = exports.NO_FILTER = exports.DEFAULT_PAGE_SIZE = exports.REQUEST_ABORTED = exports.REQUEST_SUCCESS = exports.REQUEST_FAILURE = exports.duplicateStrategy = undefined;


    var ABORT_OLD = 'ABORT_OLD';
    var DEFAULT = 'DEFAULT';
    var DROP_NEW = 'DROP_NEW';

    var duplicateStrategy = exports.duplicateStrategy = {
        ABORT_OLD: ABORT_OLD,
        DEFAULT: DEFAULT,
        DROP_NEW: DROP_NEW
    };

    var REQUEST_FAILURE = exports.REQUEST_FAILURE = 'REQUEST_FAILURE';
    var REQUEST_SUCCESS = exports.REQUEST_SUCCESS = 'REQUEST_SUCCESS';
    var REQUEST_ABORTED = exports.REQUEST_ABORTED = 'REQUEST_ABORTED';
    var DEFAULT_PAGE_SIZE = exports.DEFAULT_PAGE_SIZE = 25;
    var NO_FILTER = exports.NO_FILTER = '';

    var requestSuccess = exports.requestSuccess = function requestSuccess(action, request, dispatch, successTransform) {
        // Fallback the undefined value to null since redux can't handle the undefined as a state value
        return function () {
            var data = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : null;

            return dispatch({
                type: (0, _lodash.get)(action, 'meta.successType', REQUEST_SUCCESS),
                payload: (0, _lodash.isFunction)(successTransform) ? successTransform(data) : data,
                meta: { originalAction: action, request: request }
            });
        };
    };

    var requestAborted = exports.requestAborted = function requestAborted(action, request, dispatch) {
        return dispatch({
            type: (0, _lodash.get)(action, 'meta.abortedType', REQUEST_ABORTED),
            payload: null,
            meta: { originalAction: action, request: request }
        });
    };

    var requestFailure = exports.requestFailure = function requestFailure(action, request, dispatch) {
        return function (xhr, textStatus, error) {
            // We get the `response` from the `xhr` object, which may seem redundant since bitbucket/util/server/ajax provides
            // this already as a 4th param to the .fail() function. However, when using a different transport
            // (e.g. jQuery.ajax), this will not fly. Hence I've removed `response` as the 4th parameter,
            // and get it from `xhr` :dealwithit:
            var response = null;

            if (xhr) {
                response = xhr.responseText;
                try {
                    response = JSON.parse(xhr.responseText);
                } catch (e) {
                    // ignore
                }
            }

            return dispatch({
                type: (0, _lodash.get)(action, 'meta.failureType', REQUEST_FAILURE),
                payload: { textStatus: textStatus, error: error, response: response },
                meta: { originalAction: action, request: request }
            });
        };
    };

    var promiseRequestFailure = exports.promiseRequestFailure = function promiseRequestFailure(action, request, dispatch) {
        return function (error) {
            return dispatch({
                type: (0, _lodash.get)(action, 'meta.failureType', REQUEST_FAILURE),
                payload: { error: error },
                meta: { originalAction: action, request: request }
            });
        };
    };

    var restActorForType = exports.restActorForType = function restActorForType(type, requestBuilder) {
        var _ref = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {},
            _ref$duplicateStrateg = _ref.duplicateStrategy,
            duplicateStrategy = _ref$duplicateStrateg === undefined ? DEFAULT : _ref$duplicateStrateg,
            successTransform = _ref.successTransform,
            _ref$restTransport = _ref.restTransport,
            restTransport = _ref$restTransport === undefined ? _server.rest : _ref$restTransport;

        // The request registry cache is local to the restActor.
        // Multiple rest actors could therefore make overlapping requests to the same url,
        // without any duplicate request strategy being applied.
        // At the same time making it global would mean potentially conflicting duplicate request strategies
        // This feels like the most logical choice
        var requestsRegistry = (0, _requestsRegistry.createRequestsRegistry)();

        var restActor = function restActor(action, dispatch, state) {
            if (action.type !== type) {
                return;
            }

            var requestConfig = requestBuilder(action, state);

            if (requestConfig === null) {
                requestAborted(action, requestConfig, dispatch);

                return;
            }

            requestConfig.type = requestConfig.type || _server.method.GET;

            var maybeExistingRequest = requestsRegistry.get(requestConfig);

            if (duplicateStrategy === DROP_NEW && maybeExistingRequest) {
                return;
            }

            if (duplicateStrategy === ABORT_OLD && maybeExistingRequest && maybeExistingRequest.abort) {
                maybeExistingRequest.abort();
            }

            var request = restTransport(requestConfig);
            var isjQueryPromiseApi = typeof request.fail === 'function';

            if (isjQueryPromiseApi) {
                request.done(requestSuccess(action, requestConfig, dispatch, successTransform)).fail(requestFailure(action, requestConfig, dispatch));
            } else {
                request.then(requestSuccess(action, requestConfig, dispatch, successTransform), promiseRequestFailure(action, requestConfig, dispatch));
            }

            if (duplicateStrategy === DROP_NEW || duplicateStrategy === ABORT_OLD) {
                requestsRegistry.store(requestConfig, request);

                request[isjQueryPromiseApi ? 'always' : 'finally'](function () {
                    requestsRegistry.remove(requestConfig);
                });
            }

            return request;
        };

        //This is just to make debugging a bit easier instead of every rest actor being indistinguishable
        Object.defineProperty(restActor, 'name', { value: 'restActorFor_' + type, writable: false });

        return restActor;
    };

    /**
     * Convenience method for creating a bunch of restActors for the given types
     *
     * @param typeMap - Map of {type: requestBuilder}
     * @returns {array}
     */
    var restActorsForTypes = exports.restActorsForTypes = function restActorsForTypes(typeMap) {
        return (0, _lodash.map)(typeMap, function (requestBuilder, type) {
            return restActorForType(type, requestBuilder);
        });
    };
});