define('bitbucket/internal/util/string-replacer', ['module', 'exports', 'lodash'], function (module, exports, _lodash) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });

    var _lodash2 = _interopRequireDefault(_lodash);

    function _interopRequireDefault(obj) {
        return obj && obj.__esModule ? obj : {
            default: obj
        };
    }

    var _slicedToArray = function () {
        function sliceIterator(arr, i) {
            var _arr = [];
            var _n = true;
            var _d = false;
            var _e = undefined;

            try {
                for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) {
                    _arr.push(_s.value);

                    if (i && _arr.length === i) break;
                }
            } catch (err) {
                _d = true;
                _e = err;
            } finally {
                try {
                    if (!_n && _i["return"]) _i["return"]();
                } finally {
                    if (_d) throw _e;
                }
            }

            return _arr;
        }

        return function (arr, i) {
            if (Array.isArray(arr)) {
                return arr;
            } else if (Symbol.iterator in Object(arr)) {
                return sliceIterator(arr, i);
            } else {
                throw new TypeError("Invalid attempt to destructure non-iterable instance");
            }
        };
    }();

    function _classCallCheck(instance, Constructor) {
        if (!(instance instanceof Constructor)) {
            throw new TypeError("Cannot call a class as a function");
        }
    }

    var _createClass = function () {
        function defineProperties(target, props) {
            for (var i = 0; i < props.length; i++) {
                var descriptor = props[i];
                descriptor.enumerable = descriptor.enumerable || false;
                descriptor.configurable = true;
                if ("value" in descriptor) descriptor.writable = true;
                Object.defineProperty(target, descriptor.key, descriptor);
            }
        }

        return function (Constructor, protoProps, staticProps) {
            if (protoProps) defineProperties(Constructor.prototype, protoProps);
            if (staticProps) defineProperties(Constructor, staticProps);
            return Constructor;
        };
    }();

    var DEFAULT_REPLACER_WEIGHT = 1000;

    /**
     * For usages see string-replacer-test.js
     */

    var StringReplacer = function () {
        function StringReplacer() {
            _classCallCheck(this, StringReplacer);

            this.defaultMatcher = function (needle, haystack, startIndex) {
                return haystack.indexOf(needle, startIndex);
            };

            this.replacers = [];
            this.matcher = this.defaultMatcher;
        }

        /**
         * @param {Function} replacer - a function to that returns a Promise resolving with an array of objects used to
         * replace part(s)/all of a string.
         * @param {Number} [weight=1000] - used to rank in ascending order the sequence the replacer functions are called on
         * the string
         * @param {Function} matcher - a function to look for Jira issue keys that returns the index at which the key is
         * found.
         * @returns {Array} An array of sorted replacer functions.
         */

        /**
         * @param {string} needle - The string to find
         * @param {string} haystack - The string to search
         * @param {number} startIndex - start searching from this index
         * @returns {number} - the next index in `haystack` at which `needle` is found
         */


        _createClass(StringReplacer, [{
            key: 'registerReplacer',
            value: function registerReplacer(replacer) {
                var weight = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : DEFAULT_REPLACER_WEIGHT;
                var matcher = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : null;

                this.replacers.push({ replacer: replacer, weight: weight });
                if (typeof matcher === 'function') {
                    this.matcher = matcher;
                }

                return this.replacers.sort(function (next, prev) {
                    return next.weight - prev.weight;
                });
            }
        }, {
            key: 'occurrenceIndexes',
            value: function occurrenceIndexes(str, replacement) {
                var stringToFind = replacement.find;
                var matchMultiple = replacement.multiple !== false;
                var foundIndexes = [];
                var index = -1;

                while ((index === -1 || matchMultiple) && // this is the first attempt, or we support multiple matches
                (index = this.matcher(stringToFind, str, index + 1)) !== -1 // we found something
                ) {
                    foundIndexes.push(index);
                }

                return foundIndexes;
            }
        }, {
            key: 'addIndexes',
            value: function addIndexes(str, arrayOfReplacementArrays) {
                var _this = this;

                return _lodash2.default.flattenDeep(arrayOfReplacementArrays.map(function (_ref) {
                    var _ref2 = _slicedToArray(_ref, 2),
                        replacer = _ref2[0],
                        replacementArray = _ref2[1];

                    return replacementArray.map(function (replacement) {
                        return _this.occurrenceIndexes(str, replacement).map(function (index) {
                            return {
                                weight: replacer.weight,
                                start: index,
                                end: index + replacement.find.length,
                                replacement: replacement
                            };
                        });
                    });
                }));
            }
        }, {
            key: 'sortByWeightOrPosition',
            value: function sortByWeightOrPosition(replacementArray) {
                return replacementArray.sort(function (a, b) {
                    return a.weight - b.weight || a.start - b.start;
                });
            }
        }, {
            key: 'sortByPosition',
            value: function sortByPosition(replacementArray) {
                return replacementArray.sort(function (a, b) {
                    return a.start - b.start;
                });
            }
        }, {
            key: 'dedupe',
            value: function dedupe(str, foundReplacementsArray) {
                return foundReplacementsArray.reduce(function (deduped, replacement) {
                    var overlapped = deduped.some(function (other) {
                        return other.start <= replacement.start && other.end > replacement.start || replacement.start <= other.start && replacement.end > other.start;
                    });

                    return overlapped ? deduped : deduped.concat(replacement);
                }, []);
            }
        }, {
            key: 'replace',
            value: function replace(str, replacements, noMatchCallback) {
                if (typeof noMatchCallback === 'undefined') {
                    noMatchCallback = function noMatchCallback(str) {
                        return str;
                    };
                }

                if (!replacements.length) {
                    return noMatchCallback(str);
                }

                var results = replacements.reduce(function (memo, replacement) {
                    if (memo.pointer < replacement.start) {
                        memo.replacements.push(noMatchCallback(str.substring(memo.pointer, replacement.start)));
                    }

                    memo.replacements.push(replacement.replacement.replace);

                    return {
                        pointer: replacement.end,
                        replacements: memo.replacements
                    };
                }, { pointer: 0, replacements: [] });

                if (results.pointer < str.length) {
                    results.replacements.push(noMatchCallback(str.substring(results.pointer, str.length)));
                }

                return results.replacements.join('');
            }
        }, {
            key: 'process',
            value: function process(str, metadata, noMatchCallback) {
                var _this2 = this;

                var promises = this.replacers.map(function (_ref3) {
                    var replacer = _ref3.replacer;

                    return replacer(str, metadata).catch(function (err) {
                        // on rejection, log the error & return empty array so the resolved replacers still apply
                        console.error(err, replacer, ' failed');

                        return [];
                    });
                });

                return Promise.all(promises).then(function (arrayOfReplacementArrays) {
                    return _this2.addIndexes(str, _lodash2.default.zip(_this2.replacers, arrayOfReplacementArrays));
                }).then(this.sortByWeightOrPosition).then(function (foundReplacementsArray) {
                    return _this2.dedupe(str, foundReplacementsArray);
                }).then(this.sortByPosition).then(function (dedupedArray) {
                    return _this2.replace(str, dedupedArray, noMatchCallback);
                });
            }
        }]);

        return StringReplacer;
    }();

    exports.default = StringReplacer;
    module.exports = exports['default'];
});