define('bitbucket/internal/util/syntax-highlight/syntax-highlight', ['exports', 'codemirror', 'jquery', 'lodash', 'bitbucket/internal/util/determine-language', './highlight-codeblock', './whitespace-overlay-mode'], function (exports, _codemirror, _jquery, _lodash, _determineLanguage, _highlightCodeblock, _whitespaceOverlayMode) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.DEFAULT_LINE_ENDING = undefined;
    exports.highlightContainer = highlightContainer;
    exports.generateHighlightMarkers = generateHighlightMarkers;

    var _codemirror2 = _interopRequireDefault(_codemirror);

    var _jquery2 = _interopRequireDefault(_jquery);

    var _determineLanguage2 = _interopRequireDefault(_determineLanguage);

    function _interopRequireDefault(obj) {
        return obj && obj.__esModule ? obj : {
            default: obj
        };
    }

    /**
     * Syntax highlights a <pre><code data-language=""></code></pre>} block
     */
    var DEFAULT_LINE_ENDING = exports.DEFAULT_LINE_ENDING = '\r\n';

    /**
     * Highlight all <code>pre > code[data-language]</code> blocks in a given container
     *
     * @param {jQuery|HTMLElement} container
     * @param {Function} [callback] - An additional callback function for CodeMirror runMode
     * @return {Promise}
     */
    function highlightContainer(container, callback) {
        var promises = [];

        (0, _jquery2.default)(container).find('pre > code[data-language]').each(function (i, el) {
            var $el = (0, _jquery2.default)(el);

            // Don't run the highlighting on the suggestion code block. This would cause tabs to be changed to spaces for instance.
            if (!$el.length || el.dataset.language === 'suggestion') {
                return;
            }

            var codeblock = $el[0];
            var mode = codeblock.dataset.language;
            var text = codeblock.textContent;

            var promise = (0, _highlightCodeblock.highlightCodeblock)({
                codeblock: codeblock,
                mode: mode,
                text: text,
                callback: callback
            });

            promises.push(promise);
        });

        return Promise.all(promises).then(function () {});
    }

    /**
     * Add a marker to the collection
     * @param {array} markers
     * @param {number} lineOffset
     * @param {string} content
     * @param {string} type
     */
    var addMarker = function addMarker(markers, _ref) {
        var lineOffset = _ref.lineOffset,
            content = _ref.content,
            type = _ref.type;

        var TYPE_SEPARATOR = ' ';
        var start = lineOffset;
        var lastMarker = (0, _lodash.last)(markers);
        var hlType = type.split(TYPE_SEPARATOR).filter(function (typePart) {
            return typePart !== 'null';
        }).map(function (typePart) {
            return 'hl-' + typePart;
        }).join(TYPE_SEPARATOR);

        var isWhitespaceMarker = hlType.includes('hl-whitespace') || hlType.includes('hl-tab');

        if (!isWhitespaceMarker && lastMarker && lastMarker.type === hlType && lastMarker.end === start - 1) {
            //Merge repeated adjacent markers of the same type.
            //Especially important for the Markdown mode
            //Do not merge whitespace markers
            lastMarker.end += content.length;
        } else {
            markers.push({
                start: start,
                end: lineOffset + content.length - 1,
                type: hlType
            });
        }
    };

    /**
     * Generate the highlight markers for a given string source
     * @param {string} source
     * @param {string?} filename - The filename to infer the highlight mode from
     * @param {string?} language - The language to infer the highlight mode from (trumps filename)
     * @param {boolean?} byLine - Whether to group the markers by line or just as a single blob
     * @param {boolean?} markWhitespace - Whether to generate markers for whitespace characters (space and tab only)
     * @returns {*}
     */
    function generateHighlightMarkers(source) {
        var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
            filename = _ref2.filename,
            language = _ref2.language,
            _ref2$byLine = _ref2.byLine,
            byLine = _ref2$byLine === undefined ? false : _ref2$byLine,
            _ref2$markWhitespace = _ref2.markWhitespace,
            markWhitespace = _ref2$markWhitespace === undefined ? false : _ref2$markWhitespace;

        var modeName = '';

        if (language) {
            modeName = language;
        } else if (filename) {
            modeName = _determineLanguage2.default.fromFileInfo({
                firstLine: source.split(DEFAULT_LINE_ENDING)[0],
                path: filename
            }).name;
        }

        function runMode(cmMode) {
            var markers = [];
            var lineOffset = 0;
            var mode = markWhitespace ? {
                name: _whitespaceOverlayMode.WHITESPACE_OVERLAY,
                baseMode: cmMode.mime
            } : cmMode.mime;

            _codemirror2.default.runMode(source, mode, function (content, type, lineNumber) {
                if (!content.length) {
                    //return early to ignore any zero length tokens that CM might have incorrectly processed
                    return;
                }

                if (byLine) {
                    if (lineNumber != null) {
                        if (!markers[lineNumber]) {
                            markers[lineNumber] = [];
                            lineOffset = 0;
                        }
                        if (type) {
                            addMarker(markers[lineNumber], { lineOffset: lineOffset, content: content, type: type });
                        }
                    }

                    lineOffset += content.length;

                    return;
                }

                if (type) {
                    addMarker(markers, { lineOffset: lineOffset, content: content, type: type });
                }

                lineOffset += content.length;
            });

            return markers;
        }

        function wrmFail(reason) {
            console.warn(reason || modeName + ' could not be loaded for syntax highlighting.');
        }

        return _determineLanguage2.default.getCodeMirrorModeForName(modeName).then(runMode, wrmFail);
    }
});