define('bitbucket/internal/widget/markup-preview/markup-preview', ['module', 'exports', '@atlassian/aui', 'jquery', 'lodash', 'bitbucket/internal/util/ajax', 'bitbucket/internal/util/dom-event', 'bitbucket/internal/util/events', 'bitbucket/internal/util/navigator', 'bitbucket/internal/util/syntax-highlight/syntax-highlight', 'bitbucket/internal/util/user-keyboard-shortcuts-enabled', 'bitbucket/util/navbuilder'], function (module, exports, _aui, _jquery, _lodash, _ajax, _domEvent, _events, _navigator, _syntaxHighlight, _userKeyboardShortcutsEnabled, _navbuilder) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });

    var AJS = _interopRequireWildcard(_aui);

    var _jquery2 = _interopRequireDefault(_jquery);

    var _lodash2 = _interopRequireDefault(_lodash);

    var ajax = _interopRequireWildcard(_ajax);

    var _domEvent2 = _interopRequireDefault(_domEvent);

    var _events2 = _interopRequireDefault(_events);

    var nav = _interopRequireWildcard(_navbuilder);

    function _interopRequireDefault(obj) {
        return obj && obj.__esModule ? obj : {
            default: obj
        };
    }

    function _interopRequireWildcard(obj) {
        if (obj && obj.__esModule) {
            return obj;
        } else {
            var newObj = {};

            if (obj != null) {
                for (var key in obj) {
                    if (Object.prototype.hasOwnProperty.call(obj, key)) newObj[key] = obj[key];
                }
            }

            newObj.default = obj;
            return newObj;
        }
    }

    // Ensure that we always define a default shortcut - ie for pull-request create
    // If you modify this, please update bitbucket-plugin.xml as well
    var previewShortcutKeys = 'ctrl+shift+p';

    _events2.default.on('bitbucket.internal.DO_NOT_USE.keyboard.shortcuts.requestPreviewComment', function (keys) {
        previewShortcutKeys = keys;
        this.unbind();
    });

    /**
     * Add the ability to preview the markup content in an editor
     * @param {HTMLElement|jQuery} editor
     * @constructor MarkupPreview
     */
    function MarkupPreview(editor) {
        this.init.apply(this, arguments);
    }

    _events2.default.addLocalEventMixin(MarkupPreview.prototype);

    /**
     * Initialise the MarkupPreview
     * @param {HTMLElement|jQuery} editor
     * @param {Object} Options
     * @property {Object} lineHandle
     * @property {Object} anchor
     */
    MarkupPreview.prototype.init = function (editor, _ref) {
        var lineHandle = _ref.lineHandle,
            anchor = _ref.anchor;

        _lodash2.default.bindAll(this, 'togglePreview', 'updatePreview', 'refreshPreview', 'hidePreview', 'cancelPreview');

        var OSShortcutKeys = (0, _navigator.isMac)() ? previewShortcutKeys.replace(/ctrl/i, 'meta') : previewShortcutKeys;

        this.$editor = (0, _jquery2.default)(editor);

        this.anchor = anchor;
        this.lineHandle = lineHandle;

        this.$textarea = this.$editor.find('textarea');
        this.$previewButton = this.$editor.find('.markup-preview-button');
        this.$previewPanel = this.$editor.find('.markup-preview');
        this._destroyables = [];

        var $shortcutEls = this.$textarea.add(this.$previewButton);

        if ((0, _userKeyboardShortcutsEnabled.isUserKeyboardShortcutsEnabled)()) {
            $shortcutEls.on('keydown', [OSShortcutKeys], this.togglePreview);
        }

        this._destroyables.push({
            //This has to be done manually, not using `chainWith` because the shortcut keys param stuffs up the unbind
            destroy: function () {
                $shortcutEls.off('keydown', this.togglePreview);
            }.bind(this)
        });

        this._destroyables.push(_events2.default.chainWith(this.$previewButton).on('click', this.togglePreview));

        this._destroyables.push(_events2.default.chainWith(this.$previewPanel)
        //Don't toggle the preview if the user clicks on a link (or an image inside a link for attachment thumbnails)
        .on('click', _domEvent2.default.filterByTarget(':not(a, a > img)', this.togglePreview)));

        this._destroyables.push(_events2.default.chainWith(this.$textarea).on('input', _lodash2.default.debounce(this.refreshPreview, 100)) //prevent churning through ajax requests
        );

        this._destroyables.push({
            destroy: this.cancelPreview
        });
    };

    /**
     * Is the editor current in preview mode?
     * @returns {boolean}
     */
    MarkupPreview.prototype.isPreviewing = function () {
        return this.$editor.hasClass('previewing');
    };

    /**
     * Toggle displaying the preview
     */
    MarkupPreview.prototype.togglePreview = _domEvent2.default.preventDefault(function () {
        if (this.isPreviewing()) {
            this.cancelPreview();
        } else {
            this.showPreview();
        }
    });

    /**
     * Show the preview
     */
    MarkupPreview.prototype.showPreview = function () {
        this.$editor.addClass('previewing');
        this.$textarea.parent().spin('medium');
        this.$previewButton.text(AJS.I18n.getText('bitbucket.web.markup.edit'));
        this._request && this._request.abort();

        function onFail(xhr, reason) {
            if (reason !== 'abort') {
                //Don't hide preview on abort, it may just be because a newer request was made
                this.hidePreview();
            }
        }

        function cleanUp() {
            this.$textarea.parent().spinStop();
            this._request = null;
        }

        this._request = ajax.rest({
            type: 'POST',
            url: getPreviewUrl(),
            data: this.$textarea.val(),
            dataType: 'json'
        }).done(this.updatePreview).fail(onFail.bind(this)).always(cleanUp.bind(this));
    };

    /**
     * Refresh the preview
     */
    MarkupPreview.prototype.refreshPreview = function () {
        this.isPreviewing() && this.showPreview();
    };

    /**
     * Update the content in the preview element
     * @param {object} response
     * @param {string} response.html - the HTML for the rendered markup preview
     */
    MarkupPreview.prototype.updatePreview = function (response) {
        this.$previewPanel.html(response.html);
        this.$previewPanel.find('a').attr('target', '_blank'); //Open previewed links in a new tab
        this.$editor.addClass('loaded');
        this.$previewButton.focus();

        _events2.default.trigger('bitbucket.internal.DO_NOT_USE.feature.markup.preview', null, this.$editor, {
            isPreviewing: true
        });

        (0, _syntaxHighlight.highlightContainer)(this.$previewPanel);

        var boundResize = this.trigger.bind(this, 'resize');
        _lodash2.default.defer(boundResize); //Let the content render first
        this.$previewPanel.imagesLoaded(boundResize); //If there are images, trigger resize again once they have loaded
    };

    /**
     * Cancel the preview and hide (aborts any in progress requests for content)
     */
    MarkupPreview.prototype.cancelPreview = function () {
        this._request && this._request.abort();
        this.hidePreview();
    };

    /**
     * Hide the preview
     */
    MarkupPreview.prototype.hidePreview = function () {
        _events2.default.trigger('bitbucket.internal.DO_NOT_USE.feature.markup.preview', null, this.$editor, {
            isPreviewing: false
        });

        this.$editor.removeClass('previewing loaded');
        this.$previewButton.text(AJS.I18n.getText('bitbucket.web.markup.preview'));
        this.$textarea.focus();
        this.trigger('resize');
    };

    /**
     * Destroy the instance
     */
    MarkupPreview.prototype.destroy = function () {
        _lodash2.default.invokeMap(this._destroyables, 'destroy');
    };

    /**
     * Cached preview url builder
     * @returns {string}
     */
    var getPreviewUrl = _lodash2.default.once(function () {
        return nav.rest().markup().preview().build();
    });

    exports.default = MarkupPreview;
    module.exports = exports['default'];
});