#!/usr/bin/env bash

if [ -z "$SOURCE_H2_JAR_PATH" ]; then
  if [ -z "$SOURCE_INST_DIR" ]; then
    echo "Please provide one of the following environment variables to proceed -"
    echo "SOURCE_INST_DIR - The installation directory of the source Bitbucket version from which the H2 database file needs to be migrated"
    echo "SOURCE_H2_JAR_PATH - The location of the H2 driver JAR which is compatible with the H2 database file before migration."
    exit 1
  fi
  SOURCE_H2_JAR_PATH="$SOURCE_INST_DIR/app/WEB-INF/lib/h2-*.jar"
fi

# BIN_DIR & INST_DIR will be absolute paths, not relative
pushd $(dirname "$0") > /dev/null
BIN_DIR=$(pwd)
popd > /dev/null
INST_DIR=$(dirname "$BIN_DIR")

source "$BIN_DIR"/set-jre-home.sh &&
    source "$BIN_DIR"/set-bitbucket-home.sh &&
    source "$BIN_DIR"/set-bitbucket-user.sh

if [ -z "$BITBUCKET_SHARED_HOME" ]; then
  BITBUCKET_SHARED_HOME="$BITBUCKET_HOME"/shared
fi

if [ -z "$DB_FILE" ]; then
  PAGE_STORE_DB_FILE="$BITBUCKET_SHARED_HOME/data/db.h2.db"
  MV_STORE_DB_FILE="$BITBUCKET_SHARED_HOME/data/db.mv.db"
  if [ -f "$PAGE_STORE_DB_FILE" ]; then
    DB_FILE="$PAGE_STORE_DB_FILE"
  else
    DB_FILE="$MV_STORE_DB_FILE"
  fi
fi

if [[ "$DB_FILE" == *.mv.db ]]; then
  MV_STORE=true
else
  MV_STORE=false
fi

DB_FILE_WITHOUT_EXT="${DB_FILE%.*.db}"
TARGET_DB_FILE="$DB_FILE_WITHOUT_EXT.mv.db"
DB_LOCK_FILE="$DB_FILE_WITHOUT_EXT.lock.db"

if [ -f "$DB_LOCK_FILE" ]; then
  echo "The file $DB_LOCK_FILE is present which means the database may be in use. Please stop Bitbucket or any other clients using the $DB_FILE file and try again. If the database is not in use, manually delete the file $DB_LOCK_FILE and try again."
  exit 1
fi

if [ ! -f "$DB_FILE" ]; then
  if [ -f "$TARGET_DB_FILE" ]; then
    echo "Nothing to do; $DB_FILE has already been migrated to $TARGET_DB_FILE"
    exit 0
  else
    echo "Cannot run migration; $DB_FILE doesn't exist"
    exit 1
  fi
fi

if [ -z "$JDBC_USER" ]; then
  JDBC_USER="sa"
fi

if [ -z "$H2_JAR_PATH" ]; then
  H2_JAR_PATH="$INST_DIR/app/WEB-INF/lib/h2-*.jar"
fi

JDBC_URL="jdbc:h2:$DB_FILE_WITHOUT_EXT"
SCRIPT_FILE="h2_script.sql"
SCRIPT_JAVA_OPTS="-cp $SOURCE_H2_JAR_PATH org.h2.tools.Script -user $JDBC_USER -url $JDBC_URL;MV_STORE=${MV_STORE} -script $SCRIPT_FILE"
RUN_SCRIPT_JAVA_OPTS="-cp $H2_JAR_PATH org.h2.tools.RunScript -user $JDBC_USER -url $JDBC_URL -script $SCRIPT_FILE -showResults"

if [[ -n "$JDBC_PASSWORD" && "$JDBC_PASSWORD" != "" ]]; then
  SCRIPT_JAVA_OPTS="$SCRIPT_JAVA_OPTS -password $JDBC_PASSWORD"
  RUN_SCRIPT_JAVA_OPTS="$RUN_SCRIPT_JAVA_OPTS -password $JDBC_PASSWORD"
fi

RUN_SCRIPT_JAVA_OPTS="$RUN_SCRIPT_JAVA_OPTS -options FROM_1X"
BACKUP_FILE=$(dirname "$DB_FILE")/backup_$(basename "$DB_FILE")

echo "Creating script file..."
$JAVA_BINARY $SCRIPT_JAVA_OPTS
if [ $? -eq 0 ]; then
  echo "Script file created successfully"
  mv "$DB_FILE" "${BACKUP_FILE}"
else
  echo "Script generation failed"
  exit 1
fi

echo "Running script..."
$JAVA_BINARY $RUN_SCRIPT_JAVA_OPTS | awk '/-- [0-9]+ [+]\/- SELECT COUNT\(\*\) FROM/ {print "\t- Table "substr($7, 0, length($7)-1)" created with "$2 (($2 == 1) ? " row" : " rows")}'
if [[ $? -eq 0 && -f "$TARGET_DB_FILE" ]]; then
  echo "Run script finished, DB file generated successfully at $TARGET_DB_FILE"
  rm -rf "$SCRIPT_FILE"
  chmod u=rw,go=r "$TARGET_DB_FILE"
else
  echo "Run script failed"
  rm -rf "${DB_FILE_WITHOUT_EXT}".*.db
  mv "${BACKUP_FILE}" "$DB_FILE"
  exit 1
fi
