define('bitbucket-plugin-importer/internal/import-form/import-form', [
    '@atlassian/aui',
    'jquery',
    'lodash',
    'bitbucket/util/navbuilder',
    'bitbucket/util/server',
    'bitbucket/internal/util/analytics',
    'bitbucket/internal/util/events',
    'bitbucket/internal/util/feature-detect',
    'bitbucket/internal/util/history',
    'bitbucket/internal/util/notifications/notifications',
    '../owner-selector/owner-selector',
    '../repository-list/repository-list',
    './error-states',
    './failure-types',
    './import-state',
    './repository-selection-mode',
    './sources',
    './stages',
    'exports',
], function (
    AJS,
    $,
    _,
    navBuilder,
    server,
    analytics,
    events,
    feature,
    history,
    notifications,
    ownerSelector,
    RepositoryList,
    ErrorStates,
    FailureTypes,
    ImportState,
    RepositorySelectionMode,
    Sources,
    Stages,
    exports
) {
    'use strict';
    var $formBody;
    var repoList;

    var state;
    var utils = {};

    var importPollingTimeout;
    var baseRestUrl = navBuilder.rest('importer', 'latest').addPathComponents('external-source');

    function rerender() {
        $formBody.html(
            bitbucketPluginImporter.internal.importForm.importForm.renderWithState({
                state: state,
            })
        );
    }

    function rerenderConnect() {
        $formBody
            .find('.next-step')
            .replaceWith(
                bitbucketPluginImporter.internal.importForm.importForm.connect({ state: state })
            );
    }

    function rerenderRepoListButtons() {
        $formBody.find('.next-step').replaceWith(
            bitbucketPluginImporter.internal.importForm.importForm.repositoryListButtons({
                state: state,
            })
        );
    }

    function rerenderGithub() {
        $formBody.find('.github-specific').replaceWith(
            bitbucketPluginImporter.internal.importForm.importForm.githubSpecific({
                state: state,
            })
        );
    }

    function rerenderGitSection() {
        $formBody.find('.git-specific').replaceWith(
            bitbucketPluginImporter.internal.importForm.importForm.gitSpecific({
                state: state,
            })
        );
    }

    /**
     * Toggle if a button has a spinner displayed or not. If the spinner is displayed the button is disabled
     * @param {HTMLButtonElement} button
     * @param {boolean} toggle
     */
    function spinnerButtonToggle(button, toggle) {
        if (toggle) {
            button.setAttribute('aria-disabled', 'true');
            button.busy();
        } else {
            button.setAttribute('aria-disabled', 'false');
            button.idle();
        }
    }

    function resetLandingPageState(newSourceType) {
        var resetStateObj = {
            source: {
                type: newSourceType,
                url: null,
                error: null,
            },
            credential: {
                username: null,
                password: null,
                error: null,
            },
            githubEnterprise: false,
            validForm: false,
        };
        state = $.extend(true, state, resetStateObj);
        analytics.add('importer.form.initialised', {
            sourceType: newSourceType,
        });
    }

    function changeStage(newStage, dontAddHistory) {
        if (!dontAddHistory) {
            history.pushState(null, null, '#' + newStage);
        }

        analytics.add('importer.form.stage.changed', {
            oldStage: state.stage,
            newStage: newStage,
            sourceType: state.source.type,
            hasUrl: !(state.source.url === null),
            jobId: state.importing.jobId,
            tasks: state.importing.tasks ? state.importing.tasks.length : 0,
            githubEnterprise: state.githubEnterprise,
            possibleOwners: state.possibleOwners.length,
            repos: state.repos.length,
            repoSelectionMode: state.repoSelectionMode,
        });

        state = $.extend(true, state, {
            owner: '',
            repoSelectionMode: RepositorySelectionMode.ALL,
        });
        state.repos = [];
        repoList = null;
        state.stage = newStage;
        rerender();
    }

    function requiredFieldsPresent() {
        if (state.source.type === Sources.GIT) {
            // Every repo needs a name. The other sources provide us names when the user selects the specific
            // repos they would like to import. For Git we need to supply a name explicitly
            // Find everything after the last / that isn't trailing
            var repoName;
            var cleanSourceUrl = state.source.url.trim();

            if (state.source.url) {
                var match = cleanSourceUrl.match(/^(https?|git):\/\/[^\/]+.*\/(.+)$/);

                if (match) {
                    // Remove any trailing .git and '/' if present
                    repoName = match[2].replace(/\.git$/, '').replace(/\//, '');
                }
            }
            if (!repoName) {
                state.validForm = false;
                state.source.error = AJS.I18n.getText(
                    'bitbucket.importer.import.form.cloneUrl.error'
                );
            } else {
                state.validForm = !!state.source.url; // if credentials are filled in before clone url there might not be a url
                state.source.error = null;
                state.source.name = repoName;
                state.repos[0] = {
                    cloneUrl: cleanSourceUrl,
                    name: state.source.name,
                    description: AJS.I18n.getText(
                        'bitbucket.importer.import.form.git.repository.description',
                        cleanSourceUrl
                    ),
                    scmId: 'git',
                };
            }
            rerenderGitSection();
        } else {
            state.validForm = !!(state.credential.username && state.credential.password);

            if (state.source.type === Sources.GITHUB) {
                state.validForm = state.validForm && state.githubEnterprise === !!state.source.url;
            }
        }
    }

    utils.getOrganisations = function (baseRestRequestOptions) {
        var url = baseRestUrl
            .addPathComponents('owners')
            // request a large limit to avoid needing to make multiple requests to get owners
            // the < 100 case should be the majority case
            .withParams({ limit: 100 })
            .build();

        return server.rest(_.assign(baseRestRequestOptions, { url: url })).then(_.identity); //only care about the first param (result), stops $.when passing an array of arguments to the handler
    };

    utils.getCurrentUser = function (baseRestRequestOptions) {
        return server
            .rest(
                _.assign(baseRestRequestOptions, {
                    url: baseRestUrl.addPathComponents('user').build(),
                })
            )
            .then(_.identity);
    };

    utils.getOwner = function () {
        state.validForm = true;
        state.source.error = null;
        state.credential.error = null;

        var baseRestRequestOptions = {
            contextType: 'application/json',
            type: 'POST',
            data: {
                credential: state.credential,
                source: state.source,
            },
        };

        return $.when(
            utils.getOrganisations(baseRestRequestOptions),
            utils.getCurrentUser(baseRestRequestOptions)
        ).done(function (orgResponse, userResponse) {
            changeStage(Stages.SELECT_REPOSITORY);
            var owners = [userResponse].concat(orgResponse.values);
            state.possibleOwners = owners;
            // by default the first item is selected so use it as the owner until changed.
            state.owner = owners[0];
            updateOwnerSelector(owners);
        });
    };

    utils.importComplete = function () {
        var groupedTasks = state.importing.tasks.reduce(function (map, task) {
            var completionState;

            if (task.state === ImportState.SUCCESS) {
                completionState = task.state;
            } else if (task.failureType) {
                completionState = task.failureType;
            }
            if (!(completionState in map)) {
                map[completionState] = [];
            }
            map[completionState].push(task.externalRepoName);

            return map;
        }, {});

        var totalCount = state.importing.tasks.length;
        var successfulCount = (groupedTasks[ImportState.SUCCESS] || []).length;

        var message = bitbucketPluginImporter.internal.importForm.importForm.importCompleteMessaging(
            {
                totalCount: totalCount,
                successfulCount: successfulCount,
                nameClashRepos: groupedTasks[FailureTypes.NAME_CLASH] || [],
                authFailureRepos: groupedTasks[FailureTypes.AUTHENTICATION_FAILED] || [],
                unauthorizedRepos: groupedTasks[FailureTypes.UNAUTHORIZED] || [],
                projectName: state.project.name,
            }
        );

        if (successfulCount === totalCount) {
            notifications.addFlash(null, {
                body: message,
                close: 'manual',
            });
        } else {
            notifications.addFlash(
                AJS.I18n.getText('bitbucket.importer.import.complete.failure.title'),
                {
                    body: message,
                    type: 'error',
                    close: 'manual',
                }
            );
        }

        window.location = navBuilder.project(state.project.key).build();
    };

    function pollForProgress() {
        var $bar = $('.progress-page .progress-bar .bar');
        var url = navBuilder
            .rest('importer', 'latest')
            .addPathComponents(
                'projects',
                state.project.key,
                'import',
                'job',
                state.importing.jobId
            )
            .build();
        var completedCount = 0;
        var totalTasks = state.importing.tasks.length;
        var interval = 5000;

        function startPoll() {
            server
                .poll({
                    url: url,
                    pollTimeout: false,
                    interval: interval,
                    tick: function (response) {
                        state.importing.tasks = response.tasks;
                        completedCount = state.importing.tasks.filter(function (task) {
                            return (
                                task.state === ImportState.FAILED ||
                                task.state === ImportState.SUCCESS
                            );
                        }).length;

                        var completionProgress = (completedCount / totalTasks) * 100 || 5;
                        // always show a little bit of the polling bar so users know it's there
                        $bar.width(completionProgress + '%');

                        return completedCount === totalTasks || undefined;
                    },
                })
                .fail(function () {
                    setTimeout(startPoll, interval);
                });
        }

        startPoll();

        $bar.on(feature.transitionEndEventName(), function () {
            if (completedCount === totalTasks) {
                utils.importComplete();
            }
        });
    }

    /**
     * When a repo list row selection change, update the list of selected repositories and re-render the buttons
     * @param {boolean} checked
     * @param {Object} dataset
     */
    function repoRowChanged(checked, dataset) {
        if (checked === true) {
            // add the item to the list
            var newRepo = dataset;

            var repoExists = state.repos.some(function (repo) {
                return repo.name === newRepo.name && repo.url === newRepo.url;
            });

            if (!repoExists) {
                state.repos.push(newRepo);
            }
        } else {
            // remove the item from the list.
            state.repos = state.repos.filter(function (item) {
                return item.name !== dataset.name;
            });
        }
        rerenderRepoListButtons();
    }

    /**
     * Update the owner selector with the given values.
     * @param {Array<Object>} values
     */
    function updateOwnerSelector(values) {
        // setup owner selector
        ownerSelector({
            selector: '#repo-owners',
            searchFieldThreshold: 25,
            data: {
                results: values,
                text: 'name',
            },
        }).on('change', function (e) {
            state.owner = _.find(state.possibleOwners, { name: e.val });
            state.repos = [];
            if (state.repoSelectionMode === RepositorySelectionMode.SELECT) {
                updateRepositoryList();
            }
            rerenderRepoListButtons();
        });
    }

    /**
     * Update the repository list
     */
    function updateRepositoryList() {
        // only instantiate the repo list once per repo-selection screen lifecycle
        repoList =
            repoList ||
            new RepositoryList('#repository-list', {
                urlBuilder: baseRestUrl.addPathComponents('repos'),
                rowChangeHandler: repoRowChanged,
            });
        repoList.init({
            credential: state.credential,
            source: state.source,
            ownerName: state.owner.name,
            ownerDisplayName: state.owner.description,
        });
    }

    /**
     * Push the active source-form input values to the state.
     */
    function pushActiveFormValuesToState() {
        $formBody.find('.source-form.active input').each(function (index, field) {
            _.set(state, field.parentNode.dataset.property, field.value);
        });
    }

    //
    // UI Event Handlers
    //

    /**
     * Changes providers
     */
    function importSourceButtonHandler() {
        if (this.dataset.source === state.source.type) {
            return;
        }
        resetLandingPageState(this.dataset.source);
        rerender();
    }

    /**
     * Check the url and credential fields for a source on key up (or focus out for the git clone url)
     */
    function sourceInputFieldHandler() {
        var oldFormValid = state.validForm;
        var property = this.parentNode.dataset.property;
        _.set(state, property, this.value);
        requiredFieldsPresent();

        if (oldFormValid !== state.validForm) {
            // rerenderConnect uses replaceWith which cleans up any events attached to the removed elements
            // in order to keep the "submit" click listen around, we need to only rerender when needed, not all the time
            rerenderConnect();
        }
        events.trigger('bitbucket.internal.DO_NOT_USE.importer.form.validated');
    }

    /**
     * Switch out the github type to toggle between GH and GH:E
     */
    function githubTypeSwitchHandler() {
        var preGithubEnterprise = state.githubEnterprise;
        state.githubEnterprise = this.value === 'true';

        if (state.githubEnterprise !== preGithubEnterprise) {
            state.source.url = '';
            state.source.error = '';
            rerenderGithub();
        }
    }

    /**
     * Connect to the source with credentials.
     * @param {Event} event
     */
    function validateGivenCredentials(event) {
        event.preventDefault();

        var button = event.target;

        spinnerButtonToggle(button, true);
        pushActiveFormValuesToState();

        function errorCodeHandler(response) {
            if (response && response.responseJSON && response.responseJSON.errors) {
                return (
                    response.responseJSON.errors.filter(function (error) {
                        return error.context in ErrorStates;
                    }).length === 0
                ); // return false if there are matching errors
            }
        }

        if (state.source.type === Sources.GIT) {
            // import the given git repository
            importRepositories(button);
        } else {
            server
                .rest({
                    url: baseRestUrl.addPathComponents('missing-oauth-scopes').build(),
                    contextType: 'application/json',
                    type: 'POST',
                    data: {
                        credential: state.credential,
                        source: state.source,
                    },
                    statusCode: {
                        400: errorCodeHandler,
                        401: errorCodeHandler,
                        403: errorCodeHandler,
                    },
                })
                .done(function (additionalPermissions) {
                    if (additionalPermissions.length) {
                        state.credential.error = AJS.I18n.getText(
                            'bitbucket.importer.import.form.error.credential.morepermissions'
                        );
                        state.validForm = false;
                        rerender();
                    } else {
                        utils.getOwner().always(function () {
                            spinnerButtonToggle(button, false);
                        });
                    }
                })
                .fail(function (response) {
                    state.validForm = false;
                    state.source.error = null;
                    state.credential.error = null;
                    if (response && response.responseJSON && response.responseJSON.errors) {
                        response.responseJSON.errors.forEach(function (error) {
                            switch (error.context) {
                                case ErrorStates.AUTHENTICATION:
                                    state.credential.error = AJS.I18n.getText(
                                        'bitbucket.importer.import.form.error.credential.fail'
                                    );
                                    break;
                                case ErrorStates.INVALID_CREDENTIALS:
                                    state.credential.error = error.message;
                                    break;
                                case ErrorStates.INVALID_SOURCE:
                                    state.source.error = error.message;
                                    break;
                                case ErrorStates.EXTERNAL_REQUEST:
                                    state.validForm = true;
                                    state.source.error = AJS.I18n.getText(
                                        'bitbucket.importer.import.form.error.source'
                                    );
                                    break;
                                case ErrorStates.FORBIDDEN:
                                    // since it couldn't be checked if the form is valid set that flag to true,
                                    // so that the user can try again after changing e.g. the network environment
                                    state.validForm = true;
                                    state.source.error = error.message;
                                    break;
                            }
                        });
                    }
                    rerender();
                });
        }
    }

    /**
     * Repository selection mode changed
     * @this {HTMLElement}
     */
    function repositorySelectionModeChanged() {
        state.repos = [];
        var repoListContainerEl = document.getElementById('repos');
        state.repoSelectionMode = this.parentElement.dataset.selectMode;
        repoListContainerEl.classList.toggle(
            'hidden',
            state.repoSelectionMode === RepositorySelectionMode.ALL
        );
        if (state.repoSelectionMode === RepositorySelectionMode.SELECT) {
            updateRepositoryList();
        }
        rerenderRepoListButtons();
    }

    function submitRepos(event) {
        event.preventDefault();

        var button = event.target;
        spinnerButtonToggle(button, true);

        pushActiveFormValuesToState();
        importRepositories(button);
    }

    /**
     * Send the list of repos to be imported to the server, move to the next stage, and start polling for updates
     * @param {HTMLButtonElement} button - the "next" button that can be replaced with a spinner
     */
    function importRepositories(button) {
        var specificEndpoint = state.repos.length > 0 ? 'repos' : 'owner-repos';
        var url = navBuilder
            .rest('importer', 'latest')
            .addPathComponents('projects', state.project.key, 'import', specificEndpoint)
            .build();

        var data = {
            source: state.source,
            externalRepositories: state.repos,
            owner: state.owner,
        };

        if (state.credential.username) {
            data.credential = state.credential;
        }

        return server
            .rest({
                contextType: 'application/json',
                url: url,
                type: 'POST',
                data: data,
            })
            .done(function (response) {
                state.importing.jobId = response.jobId;
                state.importing.tasks = response.tasks;

                events.trigger('bitbucket.internal.DO_NOT_USE.importer.job.created');
                analytics.add('importer.job.created', {
                    jobId: state.importing.jobId,
                    tasks: state.importing.tasks.length,
                    sourceType: state.source.type,
                    hasUrl: !(state.source.url === null),
                });

                changeStage(Stages.IMPORTING);
                pollForProgress();
            })
            .always(function () {
                spinnerButtonToggle(button, false);
            });
    }

    function getState() {
        return state;
    }

    function setState(newState) {
        state = $.extend(true, state, newState);
    }

    //
    // Initialisation
    //

    /**
     * Initialise the module
     * @param {Object} projectJSON
     */
    function init(projectJSON, fetchTimeout) {
        state = {
            Stages: Stages,
            RepositorySelectionMode: RepositorySelectionMode,
            Sources: Sources,

            stage: Stages.LANDING_PAGE,
            source: {
                url: null,
                type: Sources.BITBUCKET_CLOUD,
                error: null,
            },
            credential: {
                username: null,
                password: null,
                error: null,
            },
            project: {
                key: projectJSON.key,
                name: null,
            },
            importing: {
                jobId: null,
                tasks: null,
            },
            githubEnterprise: false,
            validForm: false,
            possibleOwners: [],
            owner: '',
            repos: [], // the list of selected repos in the shape {name: String, cloneUrl: String, scmId: String, description: String }
            repoSelectionMode: RepositorySelectionMode.ALL,
        };
        history.replaceState(null, null, '#' + state.stage);
        state.project.name = projectJSON.name;
        importPollingTimeout = fetchTimeout * 60 * 1000;
        $formBody = $('#import-form-body');
        $formBody.html(
            bitbucketPluginImporter.internal.importForm.importForm.renderWithState({
                state: state,
            })
        );
        $formBody.on('click', '#import-source button', importSourceButtonHandler);
        $formBody.on(
            'keyup focusout',
            '.connect-to-source .active .field-group:not(.no-validate) input',
            sourceInputFieldHandler
        );
        $formBody.on(
            'focusout',
            '.connect-to-source .active .git-url input',
            sourceInputFieldHandler
        );
        $formBody.on('click', '.github-specific input.radio', githubTypeSwitchHandler);
        $formBody.on('click', '.connect-to-source button[type=submit]', validateGivenCredentials);
        $formBody.on('change', '#repositories input', repositorySelectionModeChanged);
        $formBody.on('click', '.select-repositories button[type=submit]', submitRepos);
        $formBody.on('click', 'button.cancel', function (e) {
            e.preventDefault();
            window.history.back();
        });
        history.on('popstate', function () {
            state = $.extend(true, state, {
                credential: {
                    username: null,
                    password: null,
                    error: null,
                },
                validForm: false,
            });
            changeStage(window.location.hash.split('#')[1], true);
        });
        analytics.add('importer.form.initialised', {
            sourceType: state.source.type,
        });
    }

    /**
     * @param {Object} projectJSON
     * @param {Number} fetchTimeout - how longer the server will fetch repos for before timing out, in minutes
     */
    function onReady(projectJSON, fetchTimeout) {
        $(document).ready(function () {
            init(projectJSON, fetchTimeout);
        });
    }

    exports.onReady = onReady;
    // for testing
    exports._getState = getState;
    exports._setState = setState;
    exports._init = init;
    exports._importRepositories = importRepositories;
    exports._validateGivenCredentials = validateGivenCredentials;
    exports._pushActiveFormValuesToState = pushActiveFormValuesToState;
    exports._pollForProgress = pollForProgress;
    exports._requiredFieldsPresent = requiredFieldsPresent;
    exports._resetLandingPageState = resetLandingPageState;
    exports._changeStage = changeStage;
    exports._repoRowChanged = repoRowChanged;
    exports._importSourceButtonHandler = importSourceButtonHandler;
    exports.utils = utils;
});
