'use strict';

/* global widget:false */
define('bitbucket-plugin-branch/internal/feature/branch/branch-creation/form/branch-creation-form', ['@atlassian/aui', 'jquery', 'lodash', 'bitbucket/util/navbuilder', 'bitbucket/internal/feature/repository/branch-diagram/branch-diagram', 'bitbucket/internal/feature/repository/global-repository-selector/global-repository-selector', 'bitbucket/internal/feature/repository/revision-reference-selector/revision-reference-selector', 'bitbucket/internal/model/repository', 'bitbucket/internal/model/revision-reference', 'bitbucket/internal/util/ajax', 'bitbucket/internal/util/dom-event', 'bitbucket/internal/util/events', 'bitbucket/internal/widget/simple-select/simple-select'], function (AJS, $, _2, nav, BranchDiagram, GlobalRepositorySelector, RevisionReferenceSelector, Repository, RevisionReference, ajax, domEvent, events, SimpleSelect) {
    var CUSTOM_BRANCH_TYPE = {
        id: 'CUSTOM',
        displayName: 'Custom',
        prefix: ''
    };

    var MAX_BRANCH_LENGTH = 100;

    function getCreatedBranchCheckoutUrl(repository, revisionRef) {
        return nav.project(repository.getProject()).repo(repository).withParams({
            at: revisionRef.id
        }).build();
    }

    function getCreateBranchRestUrl(repository) {
        return nav.rest('branch-utils').project(repository.getProject().getKey()).repo(repository.getSlug()).addPathComponents('branches').build();
    }

    function getBranchModelRestUrl(repository) {
        return nav.rest('branch-utils').project(repository.getProject().getKey()).repo(repository.getSlug()).addPathComponents('branchmodel').build();
    }

    function BranchCreationForm($formEl, recentRepositories, initialBranchTypes) {
        _2.bindAll(this, 'createBranch', '_noBranchTypes');

        this.$form = $formEl;
        this.repositorySelector = new GlobalRepositorySelector(this.$form.find('#repository-selector'), {
            id: 'repository-selector-dialog',
            permission: 'REPO_WRITE',
            preloadData: GlobalRepositorySelector.constructDataPageFromPreloadArray(recentRepositories)
        });
        this.$repositorySpinner = this.$form.find('.repository-spinner');
        this.repository = this.repositorySelector.getSelectedItem();
        this.branchFromSelector = new RevisionReferenceSelector(this.$form.find('.branch-from-selector'), {
            id: 'branch-from-selector-dialog',
            repository: this.repository
        });
        this.branchDiagram = new BranchDiagram(this.$form.find('.branch-diagram'));
        this.branchType = null; // init select later
        this.$branchName = this.$form.find('.branch-name');
        this.$fullBranchName = this.$form.find('.branch-and-prefix');
        this.$submitButton = this.$form.find('#create-branch-submit');
        this.$cancelButton = this.$form.find('#cancel');

        this.$branchType = this.$form.find('#branch-type');
        this.$branchTypeMenu = this.$form.find('#branch-type-menu');

        this.init(initialBranchTypes);

        this.$branchType.parent().find('label').attr('id', 'branch-type-label');
        this.$branchType.attr('aria-labelledby', 'branch-type-label').attr('role', 'combobox').attr('aria-haspopup', 'combobox').attr('aria-controls', 'branch-type-menu');
        this.$branchTypeMenu.attr('role', 'listbox');
    }

    BranchCreationForm.prototype.setActiveDescendantAria = function (self) {
        var $currentElement = $(self).attr('id');
        $('#branch-type').attr('aria-activedescendant', $currentElement);
    };

    BranchCreationForm.prototype.init = function (initialBranchTypes) {
        var self = this;

        this.$branchName.attr('maxlength', MAX_BRANCH_LENGTH);

        this.branchType = new SimpleSelect('#branch-type', '#branch-type-menu', {
            onSelect: function onSelect(prefix) {
                self._updatePrefix(prefix);
                self._updateDiagramTarget();
                self.branchFromSelector.$trigger.focus(); // focus next input
            }
        });
        this._updatePrefix(this.branchType.getSelectedValue());

        events.on('bitbucket.internal.DO_NOT_USE.feature.repository.repositorySelector.repositoryChanged', function (repository) {
            if (this === self.repositorySelector) {
                self.setRepository(repository);
            }
        });

        events.on('bitbucket.internal.DO_NOT_USE.feature.repository.revisionReferenceSelector.revisionRefChanged', function (revisionReference) {
            if (this === self.branchFromSelector) {
                self.branchDiagram.updateSourceRef(revisionReference);
            }
            self._setDisabledSubmitButton();
            self.$branchName.focus(); // focus next input
        });

        this.$branchName.on('keyup paste change cut drop', _2.debounce(function () {
            self._updateDiagramTarget();
            self._setDisabledSubmitButton();
        }, 200));

        this.$form.on('submit', domEvent.preventDefault(function () {
            if (!self.$submitButton.prop('disabled')) {
                // form can still be submitted via 'enter' even though submit button has been disabled so we should check
                self.createBranch();
            }
        }));
        this.$form.on('keydown', function (e) {
            if (domEvent.isCtrlEnter(e)) {
                e.preventDefault();
                self.$form.trigger('submit');
            }
        });

        this.$cancelButton.on('click', domEvent.preventDefault(window.history.back.bind(window.history)));

        this.branchTypesDialog = AJS.dialog2(aui.dialog.dialog2({
            titleText: AJS.I18n.getText('bitbucket.branchmodel.web.branchtypes'),
            content: bitbucketPluginBranch.internal.feature.branch.branchModel.info.branchModelInfo.branchTypeList({
                branchTypes: initialBranchTypes
            })
        }));

        $('#branch-type-dialog-trigger').on('click', function (e) {
            e.preventDefault();
            self.branchTypesDialog.show();
        });

        this.$branchTypeMenu.on('keydown', function () {
            var $currentLi = $(this).find('ul li a.active').closest('li');
            self.setActiveDescendantAria($currentLi);
        });

        this.$branchTypeMenu.find('ul li').on('click', function () {
            self.setActiveDescendantAria(this);
        });
    };

    BranchCreationForm.prototype.createBranch = function () {
        var self = this;
        this._setDisabledForm(true);

        ajax.rest({
            url: getCreateBranchRestUrl(self.repository),
            type: 'POST',
            data: {
                name: this.getBranchName(),
                startPoint: this.getBranchFrom().getId()
            },
            statusCode: {
                400: false,
                403: false,
                404: false,
                409: false,
                500: false // all handled by the .fail handler
            }
        }).done(function (revisionRef) {
            events.trigger('bitbucket.internal.DO_NOT_USE.feature.branch-creation.branchCreated', revisionRef);
            window.location = getCreatedBranchCheckoutUrl(self.repository, revisionRef);
        }).fail(function (xhr, textStatus, errorThrown, data) {
            if (data && data.errors) {
                self.$form.siblings('.aui-message').remove();
                self.$form.find('.error').remove();

                var contextualErrors = _2.filter(data.errors, function (error) {
                    return !!error.context;
                });
                _2.forEach(contextualErrors, function (error) {
                    if (error.context === 'name') {
                        self.$fullBranchName.after(widget.aui.form.fieldError({
                            message: error.message
                        }));
                    } else if (error.context === 'startPoint') {
                        self.branchFromSelector.$trigger.after(widget.aui.form.fieldError({
                            message: error.message
                        }));
                    }
                });

                var globalErrors = _2.filter(data.errors, function (error) {
                    return !error.context;
                });
                var errorBody = _2.reduce(globalErrors, function (body, error) {
                    return body + bitbucket.internal.widget.exception.exception.errorContent(error);
                }, '');

                if (errorBody.length > 0) {
                    AJS.flag({
                        type: 'error',
                        body: errorBody
                    });
                }
            }

            self._setDisabledForm(false);
        });
    };

    BranchCreationForm.prototype.setRepository = function (repository) {
        if (!repository || !repository instanceof Repository) {
            return;
        }
        this.repositorySelector.$trigger.closest('.field-group').find('.error').remove(); // Firstly, remove any repository field errors

        this.repository = repository;
        this.branchFromSelector.setRepository(this.repository);
        // Setting the repository empties out the currently selected branch,
        // but it doesn't fire the .revisionRefChanged event, so the diagram doesn't get updated.
        // Manually clear out the sourceRef from the diagram
        this.branchDiagram.updateSourceRef();
        this._updateBranchTypes();
    };

    BranchCreationForm.prototype.getBranchFrom = function () {
        return this.branchFromSelector.getSelectedItem();
    };

    BranchCreationForm.prototype.getBranchName = function () {
        //If branch-name-prefix exists in the DOM, this will use its value, otherwise .text()
        //returns an empty string if there is no user entered value (even if there is a prefix)
        return this.$branchName.val() ? this.$branchName.prev('.branch-name-prefix').text() + this._slugifyBranchName(this.$branchName.val()) : '';
    };

    BranchCreationForm.prototype._slugifyBranchName = function (branchName) {
        return branchName.replace(/\s+/g, '-');
    };

    BranchCreationForm.prototype._updateBranchTypes = function () {
        var self = this;
        this._setDisabledForm(true);
        this.$repositorySpinner.spin();
        ajax.rest({
            url: getBranchModelRestUrl(this.repository),
            type: 'GET',
            statusCode: {
                404: function _() {
                    // No branch model for this repository
                    self._setDisabledForm(false);
                    self._updatePrefix();
                    self._updateDiagramTarget();
                    self.branchFromSelector.$trigger.focus(); // focus next input

                    return false;
                },
                409: function _(xhr, textStatus, errorThrown, resp) {
                    var isEmptyRepo = _2.find(resp.errors, function (error) {
                        return error.exceptionName === 'com.atlassian.bitbucket.repository.EmptyRepositoryException';
                    });

                    if (isEmptyRepo) {
                        self.repositorySelector.$trigger.closest('.field-group').append(widget.aui.form.fieldError({
                            message: AJS.I18n.getText('bitbucket.page.branch.creation.error.repository.empty')
                        }));
                        self.repositorySelector.$trigger.prop('disabled', false).focus(); // focus same input

                        return false;
                    }
                }
            }
        }).done(function (data) {
            self._setDisabledForm(false);
            var branchTypes = data.types;

            if (branchTypes && branchTypes.length) {
                branchTypes.push(CUSTOM_BRANCH_TYPE);

                self.branchType.updateList(bitbucketPluginBranch.internal.feature.branch.branchCreation.form.branchCreationForm.branchTypesList({
                    branchTypes: branchTypes,
                    selectedBranchTypeId: self.branchType.getSelectedId() // Try to restore previous branchType
                }));

                self._updatePrefix(self.branchType.getSelectedValue());
                self._updateDiagramTarget();

                self.branchType.$trigger.closest('.field-group').removeClass('hidden');
                self.branchType.$trigger.focus(); // focus next input

                self.branchTypesDialog.$el.find('.aui-dialog2-content').html(bitbucketPluginBranch.internal.feature.branch.branchModel.info.branchModelInfo.branchTypeList({
                    branchTypes: branchTypes
                }));
            } else {
                self._noBranchTypes();
            }
        }).fail(function () {
            self._noBranchTypes();
        }).always(function () {
            self.$repositorySpinner.spinStop();
        });
    };

    BranchCreationForm.prototype._noBranchTypes = function () {
        this.$branchName.prev('.branch-name-prefix').remove();
        this.branchType.$trigger.closest('.field-group').addClass('hidden');
    };

    BranchCreationForm.prototype._updatePrefix = function (prefix) {
        this.$branchName.prev('.branch-name-prefix').remove();
        if (prefix && prefix.length) {
            var newPrefix = bitbucketPluginBranch.internal.feature.branch.branchCreation.form.branchCreationForm.branchNamePrefix({
                prefix: prefix
            });
            this.$branchName.before(newPrefix);
            this.$branchName.attr('maxlength', MAX_BRANCH_LENGTH - prefix.length);
        }
    };

    BranchCreationForm.prototype._updateDiagramTarget = function () {
        var branchName = this.getBranchName();
        var targetRef = new RevisionReference({
            id: branchName,
            displayId: branchName,
            type: RevisionReference.type.BRANCH
        });

        this.branchDiagram.updateTargetRef(targetRef);
    };

    BranchCreationForm.prototype._setDisabledSubmitButton = function (disable) {
        var branchFrom = this.getBranchFrom();
        disable = disable || !(this.getBranchName() && branchFrom && branchFrom.getId());
        this.$submitButton.prop('disabled', disable);
    };

    BranchCreationForm.prototype._setDisabledForm = function (disable) {
        this._setDisabledSubmitButton(disable);
        // jQuery takes an array of elements but not an array of jquery objects :(
        $([this.repositorySelector.$trigger[0], this.branchFromSelector.$trigger[0], this.branchType.$trigger[0], this.$branchName[0]]).prop('disabled', disable);
    };

    return BranchCreationForm;
});