define('bitbucket-plugin-repository-ref-sync/internal/feature/sync/dialog/ref-sync-dialog', [
    '@atlassian/aui',
    'bitbucket-plugin-repository-ref-sync/internal/feature/sync/common/ref-sync-common',
    'jquery',
    'lodash',
    'bitbucket/util/navbuilder',
    'bitbucket/internal/model/page-state',
    'bitbucket/internal/util/ajax',
    'bitbucket/internal/util/property',
    'bitbucket/internal/util/set-dialog-buttons-disabled',
    'bitbucket/internal/widget/submit-spinner/submit-spinner',
], function (
    AJS,
    refSyncCommon,
    $,
    _,
    nav,
    pageState,
    ajax,
    propertyUtil,
    setDialogButtonsDisabled,
    SubmitSpinner
) {
    'use strict';

    function getCloneUrl(repo) {
        return nav.project(repo.getProject()).repo(repo).clone(repo.getScmId()).buildAbsolute();
    }

    function RefSyncDialog() {
        this.action = 'merge';
    }

    RefSyncDialog.prototype.initDialog = function () {
        let self = this;
        let options;

        options = [
            {
                key: 'merge',
                styleClass: 'merge-action',
                linkText: AJS.I18n.getText('bitbucket.refsync.dialog.action.merge'),
                tooltip: AJS.I18n.getText('bitbucket.refsync.dialog.action.merge.tooltip'),
            },
            {
                key: 'discard',
                styleClass: 'discard-action',
                linkText: AJS.I18n.getText('bitbucket.refsync.dialog.action.discard'),
                tooltip: AJS.I18n.getText('bitbucket.refsync.dialog.action.discard.tooltip'),
            },
        ];

        const isGitRebaseWorkflowsEnabled =
            window.WRM.data.claim(
                'com.atlassian.bitbucket.server.bitbucket-repository-ref-sync:git.rebase.workflows.data'
            ) || false;

        if (isGitRebaseWorkflowsEnabled) {
            options.splice(1, 0, {
                key: 'rebase',
                styleClass: 'rebase-action',
                linkText: AJS.I18n.getText('bitbucket.refsync.dialog.action.rebase'),
                tooltip: AJS.I18n.getText('bitbucket.refsync.dialog.action.rebase.tooltip'),
            });
        }

        self.dialog = AJS.dialog2(
            aui.dialog.dialog2({
                id: 'ref-sync-dialog',
                content: bitbucketPluginRepositoryRefSync.internal.feature.sync.dialog.refSyncDialog.dialog(
                    {
                        ref: self.ref.toJSON(),
                        options,
                    }
                ),
                titleText: AJS.I18n.getText(
                    'bitbucket.refsync.dialog.title',
                    self.ref.getDisplayId()
                ),
                removeOnHide: true,
                footerHintContent: bitbucketPluginRepositoryRefSync.internal.feature.sync.dialog.refSyncDialog.discardWarning(),
                footerActionContent: bitbucketPluginRepositoryRefSync.internal.feature.sync.dialog.refSyncDialog.dialogFooter(),
            })
        );
        self.dialog.$el.on('click', '.cancel-button', function () {
            self.dialog.hide();
        });
        self.dialog.$el.on('click', '.confirm-button', function (e) {
            const spinner = new SubmitSpinner(e.target, 'before');
            setDialogButtonsDisabled(self.dialog, true);
            spinner.show();
            self.doSynchronization()
                .fail(function () {
                    self.dialog.hide();
                })
                .done(function () {
                    window.location.reload(true);
                });
        });
        self.dialog.$el.on('click', '.option', function (e) {
            self._setAction($(e.currentTarget));
        });
    };

    RefSyncDialog.prototype.doSynchronization = function () {
        var self = this;

        return (
            propertyUtil
                .getFromProvider('repository.' + this.action + '.sync.timeout')
                // The 'discard' option does not have a timeout. If propertyUtil fails to get a timeout then we default back to 300
                .then(
                    function (result) {
                        return $.Deferred().resolve(result);
                    },
                    function () {
                        return $.Deferred().resolve(300);
                    }
                )
                .then(function (serverTimeout) {
                    return ajax.rest({
                        url: refSyncCommon.getSyncUrl(),
                        type: 'POST',
                        timeout: serverTimeout * 1000,
                        data: {
                            action: self.action,
                            refId: self.ref.getId(),
                        },
                        statusCode: {
                            409: function (xhr, textStatus, errorThrown, data) {
                                const conflictDialog = AJS.dialog2(
                                    aui.dialog.dialog2({
                                        id: 'merge-conflict-dialog',
                                        content: bitbucketPluginRepositoryRefSync.internal.feature.sync.dialog.refSyncDialog.mergeConflictHelp(
                                            {
                                                conflicts: data.conflicts,
                                                branchName: self.ref.getDisplayId(),
                                                upstreamRemote: getCloneUrl(
                                                    pageState.getRepository().getOrigin()
                                                ),
                                                forkRemote: getCloneUrl(pageState.getRepository()),
                                            }
                                        ),
                                        titleText: AJS.I18n.getText(
                                            'bitbucket.refsync.dialog.title.merge.conflicts'
                                        ),
                                        removeOnHide: true,
                                        footerActionContent: aui.buttons.button({
                                            text: AJS.I18n.getText('bitbucket.web.button.close'),
                                            type: 'link',
                                            extraClasses: 'cancel-button',
                                        }),
                                    })
                                );
                                conflictDialog.$el.on('click', '.cancel-button', function () {
                                    conflictDialog.hide();
                                });

                                self.dialog.hide();
                                conflictDialog.show();

                                return false;
                            },
                        },
                    });
                })
        );
    };

    RefSyncDialog.prototype._setAction = function ($selectedAction) {
        this.action = $selectedAction.attr('data-option');
        const $submit = this.dialog.$el.find('.confirm-button');

        // Move the 'selected' class from the last selected element to the new one
        this.dialog.$el.find('.selected').removeClass('selected');
        $selectedAction.addClass('selected');

        $submit.text($selectedAction.find('.option-name').text());

        // Additional warning for discarding
        this.dialog.$el.find('.discard-warning').toggleClass('hidden', this.action !== 'discard');
    };

    RefSyncDialog.prototype.showFor = function (ref) {
        this.ref = ref;
        this.initDialog();
        this.dialog.show();
    };

    return RefSyncDialog;
});
