define('bitbucket-plugin-comment-likes/internal/feature/comments/likes/web-panel', [
    'bitbucket/util/state',
    'bitbucket/internal/feature/comments/comment-async-web-panel',
    'bitbucket-plugin-comment-likes/internal/feature/comments/likes/controller',
    'exports',
], function (pageState, asyncWebPanel, CommentLikes, exports) {
    'use strict';

    /**
     * A easily rendered set of like information.
     * @typedef {Object} LikeInfo
     * @property {boolean} canLike - If the user can like/unlike this comment.
     * @property {number} likerCount - The number of people who like this comment.
     * @property {string} otherLiker - The first person who liked this comment other than the current user.
     * @property {Array<StashUser>} otherLikers - A list of people other than the current user who like this comment.
     * @property {boolean} userHasLiked - If the user likes this comment.
     */

    /**
     * Convert the like information into an easily rendered form.
     *
     * @param {Object} context
     * @param {Comment} context.comment The comment that likes will be added to.
     * @returns {LikeInfo}
     */
    function extractLikeInfo(context) {
        var comment = context.comment;

        var likedBy = comment.properties && comment.properties.likedBy;
        var currentUserSlug = pageState.getCurrentUser() && pageState.getCurrentUser().slug;

        var state = {
            userHasLiked: false,
            otherLiker: null,
            likerCount: 0,
            otherLikers: [],
            canLike: comment.author.slug !== currentUserSlug,
        };

        if (likedBy) {
            state.userHasLiked = likedBy.total > 0 && likedBy.likers[0].slug === currentUserSlug;
            if (state.userHasLiked) {
                state.otherLiker = likedBy.total > 1 ? likedBy.likers[1] : null;
            } else {
                state.otherLiker = likedBy.total > 0 ? likedBy.likers[0] : null;
            }
            state.likerCount = likedBy.total;
            state.otherLikers = state.userHasLiked ? likedBy.likers.slice(1) : likedBy.likers;
        }

        return state;
    }

    /**
     * Returns if the user can like this comment.
     *
     * @param {Object} context
     * @returns {boolean} If the user can like the comment.
     */
    function buttonPanelCondition(context) {
        return extractLikeInfo(context).canLike;
    }

    /**
     * Renders the info webPanel using comment-async-web-panel.
     *
     * @param {Object} context
     * @returns {string} the placeholder HTML.
     */
    function infoPanel(context) {
        return asyncWebPanel.getWebPanelEl(function ($el) {
            new CommentLikes({
                commentId: context.comment.id,
                likes: extractLikeInfo(context),
                $infoPlaceholder: $el,
            });
        });
    }

    exports.contextProvider = extractLikeInfo;
    exports.buttonPanelCondition = buttonPanelCondition;
    exports.infoPanel = infoPanel;
});
