define('bitbucket-plugin-ref-restriction/internal/feature/dialog/branch-permissions-dialog', [
    '@atlassian/aui',
    'jquery',
    'lodash',
    'eve',
    'bitbucket/util/navbuilder',
    'bitbucket/internal/feature/user/user-and-group-and-access-key-multi-selector/user-and-group-and-access-key-multi-selector',
    'bitbucket/internal/model/page-state',
    'bitbucket/util/state',
    'bitbucket/internal/enums',
    'bitbucket/internal/util/ajax',
    'bitbucket/internal/util/dom-event',
    'bitbucket/internal/util/function',
    'bitbucket/internal/util/scope-type',
    '../../model/ref-matcher',
    '../../model/ref-restriction',
], function (
    AJS,
    $,
    _,
    eve,
    nav,
    UserAndGroupAndAccessKeyMultiSelector,
    modelPageState,
    pageState,
    enums,
    ajax,
    domEventUtil,
    fn,
    scopeTypeUtil,
    RefMatcher,
    RefRestriction
) {
    var ScopeType = enums.ScopeType;

    /**
     * Enum for the available restriction types
     * @type {{READ_ONLY: string, FAST_FORWARD_ONLY: string, PULL_REQUEST_ONLY: string, NO_CREATES: string, NO_DELETES: string}}
     */
    var RESTRICTION_TYPE = {
        READ_ONLY: 'read-only',
        FAST_FORWARD_ONLY: 'fast-forward-only',
        PULL_REQUEST_ONLY: 'pull-request-only',
        NO_CREATES: 'no-creates',
        NO_DELETES: 'no-deletes',
    };

    /**
     * Mapping for the checkbox id to the corresponding restriction type
     * @type {{prevent-changes-checkbox: string, prevent-force-push-checkbox: string, prevent-merge-checkbox: string, prevent-creation-checkbox: string, prevent-deletion-checkbox: string}}
     */
    var CHECKBOX_ID = {
        'prevent-changes-checkbox': 'read-only',
        'prevent-force-push-checkbox': 'fast-forward-only',
        'prevent-merge-checkbox': 'pull-request-only',
        'prevent-creation-checkbox': 'no-creates',
        'prevent-deletion-checkbox': 'no-deletes',
    };

    /**
     * Enum for the available pickers
     * @type {{BRANCH: string, PATTERN: string, MODEL: string}}
     */
    var PICKER_TYPES = {
        BRANCH: 'BRANCH',
        PATTERN: 'PATTERN',
        MODEL: 'MODEL',
    };

    /**
     * @typedef {Object} DialogOptions
     * @param {Object} permittedEntities - a map of restriction type to an array of permitted entities
     * @param {boolean} editing - true if the dialog is in edit mode, false otherwise
     * @param {Object} prevent - maps the various 'prevents' to a permission ID (or undefined if none exists)
     * @param {RefMatcher} matcher
     */

    /**
     * @param {Object} options
     * @param {Object} options.refRestrictions
     * @param {DialogOptions} options.dialogOptions
     * @param {ScopeType} options.scopeType
     * @constructor
     */
    function BranchPermissionDialog(options) {
        _.bindAll(this, _.functions(this.constructor.prototype));
        this._scopeType = options.scopeType;
        this.currentScopeNavMethod = scopeTypeUtil.scopeNavAndStateMethod(this._scopeType);
        this._dialogOptions = options.dialogOptions || {};
        this._dialogOptions.scopeType = this._scopeType;
        this._dialog = AJS.dialog2(
            bitbucketPluginRefRestriction.internal.feature.dialog.branchPermissionsDialog.permissionDialog(
                this._dialogOptions
            )
        );
        this.refRestrictions = options.refRestrictions || {};
        this._dialog.$el.on('change', 'input, #ref-selector-trigger', this.validate);
        this._dialog.$el.on(
            'change',
            '#restriction-types-checkbox-field input',
            this.showHideEntityFields
        );
        this._dialog.$el.on(
            'click',
            '.error-message-edit-link',
            this.openEditDialogForCurrentMatcher
        );
    }

    /**
     * @param selector
     * @returns {jQuery} jQuery object from the BranchPermissionsDialog matching the selector
     */
    BranchPermissionDialog.prototype.$ = function (selector) {
        return this._dialog.$el.find(selector);
    };

    /**
     * Shows and initializes the BranchPermissionsDialog
     * @param callback The function to call once the dialog is saved
     */
    BranchPermissionDialog.prototype.show = function (callback) {
        this.callback = callback;
        this._dialog.show();
        this.init();
    };

    /**
     * Initializes all the elements on the BranchPermissionDialog
     */
    BranchPermissionDialog.prototype.init = function () {
        getBranchModels(this.currentScopeNavMethod, this._scopeType).fail(this.disableBranchModel);
        this.initMatcherTypeSelector();
        this.initBranchSelector();
        this.initButtons();
        this.initCheckboxes();
        this.initKeypress();
        this.validate();
    };

    /**
     * Attaches the event handlers to the save and cancel buttons in the Dialog's footer
     */
    BranchPermissionDialog.prototype.initButtons = function () {
        this.$('#save-permission-button').on('click', this.save);
        this.$('#cancel-permission-button').on('click', this.hide);
    };

    /**
     * Attaches the event handlers to the branch ref selector input
     */
    BranchPermissionDialog.prototype.initBranchSelector = function () {
        var $branchRefInputField = this.$('#branch-ref-input-field');
        $branchRefInputField.on('click', '.remove-link', this.validate);
        // Focus either the branch pattern input field or the button for the subsequent select2
        $branchRefInputField.find('input, button').focus();
        $branchRefInputField.on('keyup', this.validate);
    };

    BranchPermissionDialog.prototype.initBranchModelSelector = function () {
        /**
         * Return a string containing the HTML to be used for the custom select row
         * @param {Select2Item} item - a select2 row item
         * @returns {string}
         */
        function tplFunction(item) {
            var $optEl = $(item.element);

            return bitbucketPluginRefRestriction.internal.feature.dialog.branchPermissionsDialog.branchModelSelectorRowTemplate(
                {
                    name: $optEl.text(),
                    pattern: $optEl.data('display-name'),
                    isBranch: $optEl.data('is-branch') === true,
                }
            );
        }

        var $branchModelSelector = this.$('#branch-model-selector');
        $branchModelSelector.auiSelect2({
            placeholder: AJS.I18n.getText(
                'bitbucket.web.branch.permissions.dialog.model.select.placeholder'
            ),
            formatResult: tplFunction,
            formatSelection: tplFunction,
            // Ensures the search box is hidden for this selector
            minimumResultsForSearch: Infinity,
        });
        $branchModelSelector.on('change', this.validate);
    };

    BranchPermissionDialog.prototype.initCheckboxes = function () {
        this.showHideEntityFields();
        this.refreshCheckboxLabels();
        this.$('input[type="checkbox"]').on('change', this.refreshCheckboxLabels.bind(this));
    };

    /**
     * Adds a keypress listener for the dialog
     * Available keyboard shortcuts:
     * * Cmd+Enter - Submit
     */
    BranchPermissionDialog.prototype.initKeypress = function () {
        var self = this;
        this._dialog.$el.on('keydown', function (e) {
            if (domEventUtil.isCtrlEnter(e)) {
                e.preventDefault();
                self.$('#save-permission-button').click();
            }
        });
    };

    /**
     * Attaches the event handlers to the matcher type fields or
     * hides the fields if the dialog is opened in edit mode
     */
    BranchPermissionDialog.prototype.initMatcherTypeSelector = function () {
        var $matcherType = this.$('input[name="matcherType"]');
        var self = this;

        if (this._dialogOptions.editing) {
            $matcherType.parents('.radio').remove();

            return;
        }

        $matcherType.on('change', function (e) {
            self.updateBranchField({
                scopeType: self._scopeType,
                branchType: $(this).val(),
                branchRef:
                    self._dialogOptions &&
                    self._dialogOptions.matcher &&
                    self._dialogOptions.matcher.id,
            });
            e.target.focus();
        });

        this.updateBranchField({
            scopeType: this._scopeType,
            branchType: $matcherType.val(),
            branchRef:
                self._dialogOptions &&
                self._dialogOptions.matcher &&
                self._dialogOptions.matcher.id,
        });
    };

    /**
     * Attaches the event handlers to the entity selector and ensures only licensed users and users with
     * write permission to the repository will show up.
     * @param {jQuery} $field - the field for which to initialize the entity selector
     */
    BranchPermissionDialog.prototype.initEntitySelector = function ($field) {
        // add user selector params to filter out unlicensed/unauthorized users
        var multiSelectorOptions = {
            urls: {},
            urlParams: {
                user: {
                    'permission.1': 'LICENSED_USER', // only licensed users
                    'permission.2': 'REPO_WRITE', // only users with WRITE to the repository, otherwise they can't push anyway
                },
                accessKey: {
                    permission: 'REPO_WRITE', // only access keys with WRITE to the repository
                    effective: true,
                },
            },
        };

        if (this._scopeType === ScopeType.REPOSITORY) {
            multiSelectorOptions.urls.accessKey = function () {
                return nav
                    .rest('keys', 'latest')
                    .repository(pageState.getRepository())
                    .addPathComponents('ssh')
                    .build();
            };
            multiSelectorOptions.urlParams.user[
                'permission.2.repositoryId'
            ] = modelPageState.getRepository().getId();
        } else {
            multiSelectorOptions.urls.accessKey = function () {
                return nav
                    .rest('keys', 'latest')
                    .project(pageState.getProject())
                    .addPathComponents('ssh')
                    .build();
            };
            multiSelectorOptions.urlParams.user['permission.2'] = 'PROJECT_WRITE'; // only users with WRITE to the project
            multiSelectorOptions.urlParams.user[
                'permission.2.projectId'
            ] = modelPageState.getProject().getId();
            multiSelectorOptions.urlParams.accessKey.permission = 'PROJECT_WRITE'; // only access keys with WRITE to the PROJECT
        }

        var forField = $field.attr('for');
        var $entitySelector = $field.find('input[type="text"]');

        if (!this.entitySelectors) {
            this.entitySelectors = {};
        }
        this.entitySelectors[forField] = new UserAndGroupAndAccessKeyMultiSelector(
            $entitySelector,
            multiSelectorOptions
        );

        var permittedEntities =
            this._dialogOptions.permittedEntities &&
            this._dialogOptions.permittedEntities[CHECKBOX_ID[forField]];

        if (permittedEntities) {
            this.entitySelectors[forField].setSelectedItems({
                users: permittedEntities.users || [],
                groups: permittedEntities.groups || [],
                accessKeys: permittedEntities.accessKeys || [],
            });
        }
    };

    /**
     * Closes the branch ref selector.
     * Needs to be here because opening the branch ref selector and
     * clicking a (different) select2 trigger will not close the branch ref selector dialog.
     */
    BranchPermissionDialog.prototype.ensureBranchSelectorIsClosed = function () {
        $('.searchable-selector-dialog').hide();
    };

    BranchPermissionDialog.prototype.branchSelectionNotEmpty = function () {
        return this.getCurrentMatcher() !== null;
    };

    /**
     * Determine if any entity selector field has changed from the initial values the dialog was opened with
     * @returns {boolean} true if the values in any entity selector have changed, false otherwise
     */
    BranchPermissionDialog.prototype.haveAccessGrantsChanged = function () {
        var containTheSameValues = function (arr1, arr2) {
            return $(arr1).not(arr2).length === 0 && $(arr2).not(arr1).length === 0;
        };

        if (this.entitySelectors) {
            var self = this;

            return !_.every(this.entitySelectors, function (entitySelector, key) {
                var fieldValues = entitySelector.getSelectedItems();
                var existingValues =
                    (self._dialogOptions.permittedEntities &&
                        self._dialogOptions.permittedEntities[key]) ||
                    {};

                if (
                    containTheSameValues(fieldValues.groups, existingValues.groups) &&
                    containTheSameValues(fieldValues.users, existingValues.users) &&
                    containTheSameValues(fieldValues.accessKeys, existingValues.accessKeys)
                ) {
                    return false;
                }

                return true;
            });
        } else if (
            !this._dialogOptions.permittedEntities ||
            Object.keys(this._dialogOptions.permittedEntities).length === 0
        ) {
            return false;
        }

        return true;
    };

    /**
     * @returns {boolean} true if at least one permission field (entity selector or a checkbox) has changed from the initial value
     */
    BranchPermissionDialog.prototype.oneOrMorePermissionsChanged = function () {
        if (this._dialogOptions.editing) {
            // In edit mode we always want to have the save button enabled, regardless of if a field has changed.
            return true;
        }

        return this.haveAccessGrantsChanged() || this.isAtLeastOneCheckboxChanged();
    };

    BranchPermissionDialog.prototype.refreshCheckboxLabels = function () {
        function getTextFor(id, checked) {
            if (checked) {
                switch (id) {
                    case 'prevent-changes-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.changes.checked'
                        );
                    case 'prevent-force-push-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.force.checked'
                        );
                    case 'prevent-merge-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.merge.checked'
                        );
                    case 'prevent-deletion-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.deletion.checked'
                        );
                    case 'prevent-creation-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.creation.checked'
                        );
                }
            } else {
                switch (id) {
                    case 'prevent-changes-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.changes'
                        );
                    case 'prevent-force-push-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.force'
                        );
                    case 'prevent-merge-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.merge'
                        );
                    case 'prevent-deletion-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.deletion'
                        );
                    case 'prevent-creation-checkbox':
                        return AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.prevent.creation'
                        );
                }
            }

            return '';
        }

        Object.keys(CHECKBOX_ID).forEach(function (id) {
            $('label[for="' + id + '"]').text(getTextFor(id, document.getElementById(id).checked));
        });
    };

    /**
     * @returns {boolean} true if at least one checkbox has changed from the initial value
     */
    BranchPermissionDialog.prototype.isAtLeastOneCheckboxChanged = function () {
        var checkedFields = 0;
        var existingRestrictions = 0;
        this.$('input[type="checkbox"]').each(function (i, checkbox) {
            if (checkbox.checked) {
                checkedFields = checkedFields + i + 1;
            }
        });

        if (this._dialogOptions.prevent) {
            var iteration = 0;
            _.forEach(this._dialogOptions.prevent, function (refExists) {
                if (refExists) {
                    existingRestrictions = existingRestrictions + iteration + 1;
                }
                iteration++;
            });
        }

        return checkedFields !== existingRestrictions;
    };

    /**
     * Sets the disabled attribute on the submit button
     * @param isDisabled {boolean} to set the disabled attribute to
     */
    BranchPermissionDialog.prototype.disableSubmit = function (isDisabled) {
        this.$('#save-permission-button').prop('disabled', isDisabled);
    };

    /**
     * @returns {boolean} true if the form, in it's current state is valid and can be submitted, false otherwise
     */
    BranchPermissionDialog.prototype.checkIfCanSubmit = function () {
        return this.branchSelectionNotEmpty() && this.oneOrMorePermissionsChanged();
    };

    /**
     * Check if the selected permission ref does not exist yet.
     * In edit mode always returns true, as we have to assume the permission exists.
     * If the permission does exist, renders an error message with a link to edit the existing permission.
     * @returns {boolean} true if no permission matching the selected type and ref exists, false otherwise
     */
    BranchPermissionDialog.prototype.selectedBranchPermissionDoesNotExist = function () {
        if (this._dialogOptions.editing) {
            //In edit mode we need to have a branch with existing permissions, therefore we skip this validator
            return true;
        }

        var selectedRefMatcher = this.getCurrentMatcher();
        var exists = false;
        var scopeType = this._scopeType;

        if (selectedRefMatcher) {
            exists = this.refRestrictions.any(function (restrictedRef) {
                var sameScope = scopeType === restrictedRef.getScope().type;
                var refMatcher = restrictedRef.getMatcher();

                return sameScope && refMatcher.equals(selectedRefMatcher);
            });
        }

        this.resetErrors();
        if (exists) {
            var errorContent = aui.message.error({
                content: bitbucketPluginRefRestriction.internal.feature.dialog.branchPermissionsDialog.duplicateError(
                    {
                        errorMatcher: selectedRefMatcher.getDisplayId(),
                    }
                ),
            });

            $(errorContent).insertBefore(this.$('.branch-permissions-dialog-form'));
        }

        return !exists;
    };

    /**
     * Validates the form in the dialog.
     * If the form is valid, enables the submit button, if not disables it.
     */
    BranchPermissionDialog.prototype.validate = function () {
        this.disableSubmit(!this.isValid());
    };

    /**
     * Check if the form, in it's current state is valid.
     * If the form is not valid, disable the submit button.
     * @returns {boolean} true if the form is valid, false otherwise
     */
    BranchPermissionDialog.prototype.isValid = function () {
        var checks = [this.selectedBranchPermissionDoesNotExist, this.checkIfCanSubmit];

        return checks.every(fn.lazyApply());
    };

    BranchPermissionDialog.prototype.hide = function () {
        this._dialog.hide();
    };

    BranchPermissionDialog.prototype.showSpinner = function () {
        this.$('.spinner-wrapper').spin();
    };

    BranchPermissionDialog.prototype.stopSpinner = function () {
        this.$('.spinner-wrapper').spinStop();
    };

    /**
     * Renders any errors returned in the AJAX response
     * @param response
     * @param textStatus
     * @param errorThrown
     * @param data The returned content
     * @returns {boolean}
     */
    BranchPermissionDialog.prototype.responseErrorHandler = function (
        response,
        textStatus,
        errorThrown,
        data
    ) {
        if (data.errors && data.errors.length) {
            this.renderErrors(data.errors);

            return false;
        }
    };

    /**
     * Removes the "Branch model" select option from the selector
     */
    BranchPermissionDialog.prototype.disableBranchModel = function () {
        this.$('#matcherType-model').parents('.radio').remove();
    };

    /**
     * Queries the REST API for the current branch model.
     * @returns {Ajax} which will return an {Array} of {Object}s if successful
     */
    var getBranchModels = _.once(function getBranchModels(currentScopeNavMethod, scopeType) {
        var urlBuilder = nav
            .rest('branch-utils', 'latest')
            [currentScopeNavMethod.nav](pageState[currentScopeNavMethod.state]())
            .addPathComponents('branchmodel');

        if (scopeType === ScopeType.PROJECT) {
            urlBuilder = urlBuilder.addPathComponents('configuration');
        }
        var reject = function () {
            return $.Deferred().reject();
        };

        return ajax
            .rest({
                url: urlBuilder.build(),
                statusCode: {
                    // if there is an error getting the branch model config, reject here so it gets disabled as a choice
                    400: reject,
                    404: reject,
                    409: reject,
                },
            })
            .then(function (response) {
                var branchModels = [];

                if (response.development) {
                    branchModels.push({
                        id: 'development',
                        name: AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.model.development'
                        ),
                        pattern:
                            response.development.refId ||
                            response.development.displayId ||
                            AJS.I18n.getText(
                                'bitbucket.web.branch.permissions.model.default.branch'
                            ),
                        isBranch: true,
                    });
                }

                if (response.production) {
                    branchModels.push({
                        id: 'production',
                        name: AJS.I18n.getText(
                            'bitbucket.web.branch.permissions.dialog.model.production'
                        ),
                        pattern:
                            response.production.refId ||
                            response.production.displayId ||
                            AJS.I18n.getText(
                                'bitbucket.web.branch.permissions.model.default.branch'
                            ),
                        isBranch: true,
                    });
                }

                if (response.types) {
                    response.types.forEach(function (type) {
                        branchModels.push({
                            id: type.id,
                            name: type.displayName,
                            pattern: type.prefix,
                            isBranch: false,
                        });
                    });
                }

                return branchModels;
            });
    });

    BranchPermissionDialog.prototype.initBranchSelectorInput = function ($inputField) {
        const branchSelectorInput = $inputField.find('.branch-selector-input')[0];

        if (branchSelectorInput) {
            const { id, ...data } = branchSelectorInput.dataset;
            eve('bitbucket.internal.DO_NOT_USE.widget.branchselector.inputAdded', null, id, data);
        }
    };

    /**
     * Replaces the branch ref selector with the specified template content.
     * Validates the form, to ensure the submit button is disabled with the new, empty field, being in the form.
     * @param branchSelectionField template function to a branch ref selector field
     */
    BranchPermissionDialog.prototype.updateBranchField = function (opts) {
        var updateDOMFields = function (options) {
            var $inputField = $('#branch-ref-input-field');
            $inputField.html(
                bitbucketPluginRefRestriction.internal.feature.dialog.branchPermissionsDialog.branchRefField(
                    options
                )
            );
            this.initBranchSelectorInput($inputField);
            this.initBranchModelSelector();
            $inputField.find('input, button, .branch-model-selector a').focus();

            // detach this item from the DOM and move it to the radio that triggered it
            var $callee = $('input[value="' + opts.branchType + '"]').parents('.radio');
            $callee.append($inputField);

            this.showHideEntityFields();
            this.validate();
        };

        if (opts.branchType === PICKER_TYPES.MODEL) {
            getBranchModels(this.currentScopeNavMethod, this._scopeType)
                .then(function (branchModels) {
                    opts.branchModels = branchModels;
                })
                .done(updateDOMFields.bind(this, opts));
        } else {
            updateDOMFields.bind(this, opts)();
        }
    };

    /**
     * Removes all errors from the dialog.
     */
    BranchPermissionDialog.prototype.resetErrors = function () {
        this.$('.error').remove();
    };

    /**
     * Renders errors to the top of the dialog content.
     * @param {Array<Object>} containing errors to render
     */
    BranchPermissionDialog.prototype.renderErrors = function (errors) {
        this.resetErrors();
        if (errors.responseJSON) {
            errors = errors.responseJSON.errors;
        }
        var errorHTMLMessages = [];
        var unknownError = AJS.I18n.getText(
            'bitbucket.web.branch.permissions.dialog.error.unknown'
        );

        if (!Array.isArray(errors)) {
            errorHTMLMessages.push(
                aui.message.error({
                    content: unknownError,
                })
            );
        } else {
            errorHTMLMessages = errors.map(function (error) {
                if (!error.message) {
                    error = {
                        message: unknownError,
                    };
                }

                return aui.message.error({
                    content: _.escape(error.message),
                });
            });
        }

        $(errorHTMLMessages.join('')).insertBefore(this.$('.branch-permissions-dialog-form'));
    };

    /**
     * Finds permissions matching the specified branch type and branch value
     * @param {RefMatcher} matcher
     * @returns {Array<RefRestriction>} permissions
     */
    BranchPermissionDialog.prototype.getPermissionsFor = function (matcher) {
        var permissions = [];
        var self = this;
        this.refRestrictions.each(function (restrictedRef) {
            var refMatcher = restrictedRef.getMatcher();

            if (matcher.equals(refMatcher) && restrictedRef.getScope().type === self._scopeType) {
                permissions.push(restrictedRef);
            }
        });

        return permissions;
    };

    BranchPermissionDialog.prototype.openEditDialogForCurrentMatcher = function () {
        this.openEditDialogFor(this.getCurrentMatcher());
    };

    /**
     * Hides the current dialog and opens a new dialog in edit mode for the specified branch type and branch value
     * @param {RefMatcher} matcher
     */
    BranchPermissionDialog.prototype.openEditDialogFor = function (matcher) {
        this.hide();
        new BranchPermissionDialog({
            refRestrictions: this.refRestrictions,
            dialogOptions: BranchPermissionDialog.getDialogOptionsFor(matcher, null, this),
            scopeType: this._scopeType,
        }).show(this.callback);
    };

    /**
     * Creates and returns a custom BranchPermissionDialog options {Object}
     * @param {RefMatcher} matcher
     * @param {Array<RefRestriction>} permissions
     * @param {BranchPermissionDialog} context
     * @returns {DialogOptions}
     */
    BranchPermissionDialog.getDialogOptionsFor = function (matcher, permissions, context) {
        if (!permissions) {
            permissions = context.getPermissionsFor(matcher);
        }

        var permittedEntities = {};
        var permissionChanges;
        var permissionForce;
        var permissionMerge;
        var permissionCreation;
        var permissionDeletion;

        function toPermittedEntity(permission) {
            return {
                id: permission.getId(),
                users: permission.getUsers(),
                groups: permission.getGroups(),
                accessKeys: permission.getAccessKeys(),
            };
        }

        permissions.forEach(function (permission) {
            switch (permission.getType()) {
                case RESTRICTION_TYPE.READ_ONLY:
                    permissionChanges = permission.getId();
                    permittedEntities[RESTRICTION_TYPE.READ_ONLY] = toPermittedEntity(permission);
                    break;
                case RESTRICTION_TYPE.FAST_FORWARD_ONLY:
                    permissionForce = permission.getId();
                    permittedEntities[RESTRICTION_TYPE.FAST_FORWARD_ONLY] = toPermittedEntity(
                        permission
                    );
                    break;
                case RESTRICTION_TYPE.PULL_REQUEST_ONLY:
                    permissionMerge = permission.getId();
                    permittedEntities[RESTRICTION_TYPE.PULL_REQUEST_ONLY] = toPermittedEntity(
                        permission
                    );
                    break;
                case RESTRICTION_TYPE.NO_CREATES:
                    permissionCreation = permission.getId();
                    permittedEntities[RESTRICTION_TYPE.NO_CREATES] = toPermittedEntity(permission);
                    break;
                case RESTRICTION_TYPE.NO_DELETES:
                    permissionDeletion = permission.getId();
                    permittedEntities[RESTRICTION_TYPE.NO_DELETES] = toPermittedEntity(permission);
                    break;
            }
        });

        return {
            editing: permissions.length > 0,
            matcher: matcher ? matcher.toJSON() : RefMatcher.branch(),
            permittedEntities: permittedEntities,
            prevent: {
                changes: permissionChanges,
                creation: permissionCreation,
                force: permissionForce,
                merge: permissionMerge,
                deletion: permissionDeletion,
            },
        };
    };

    /**
     * @returns {RefMatcher?}
     */
    BranchPermissionDialog.prototype.getCurrentMatcher = function () {
        if (this._dialogOptions.editing) {
            return this._dialogOptions.matcher;
        }
        switch (this.$('input[name="matcherType"]:checked').val()) {
            case PICKER_TYPES.BRANCH:
                var selectedRef = null;

                if (this._scopeType === ScopeType.REPOSITORY) {
                    // For repos get the revision ref from the picker
                    selectedRef = this.$('#ref-selector-trigger .name').data('revision-ref');
                } else {
                    // For projects get the branch value and use it as a ref.
                    var branchVal = this.$('#branch-name-field').val();

                    if (!branchVal || _.includes(branchVal, ' ')) {
                        return null;
                    }
                    selectedRef = {
                        id: branchVal,
                        displayId: branchVal,
                    };
                }

                return selectedRef ? new RefMatcher().branch(selectedRef) : null;
            case PICKER_TYPES.PATTERN:
                var branchPatternFieldValue = this.$('#branch-pattern-field').val();

                if (branchPatternFieldValue && branchPatternFieldValue.trim() !== '') {
                    return new RefMatcher().pattern(branchPatternFieldValue);
                }

                return null;
            case PICKER_TYPES.MODEL:
                var $branchModelField = this.$('#branch-model-selector');
                var branchModelFieldValue = $branchModelField.val();
                var branchModelFieldValueIsBranch = $branchModelField
                    .find(':selected')
                    .data('is-branch');

                if (branchModelFieldValue && branchModelFieldValue.trim() !== '') {
                    return new RefMatcher().model(
                        branchModelFieldValue,
                        branchModelFieldValueIsBranch
                    );
                }

                return null;
        }
    };

    BranchPermissionDialog.prototype.getRestPermissionFor = function (
        refRestrictionId,
        checkboxId
    ) {
        var accessGrants = this.entitySelectors[checkboxId].getSelectedItems();
        var permission = null;

        if (refRestrictionId) {
            permission = this.refRestrictions.get(refRestrictionId);
        }

        if (!permission) {
            var scopeId =
                this._scopeType === ScopeType.REPOSITORY
                    ? modelPageState.getRepository().id
                    : modelPageState.getProject().id;
            var refMatcher = this.getCurrentMatcher();
            permission = new RefRestriction().setDetails({
                matcher: refMatcher,
                scope: { type: this._scopeType, resourceId: scopeId },
                type: CHECKBOX_ID[checkboxId],
            });
        }

        permission.setDetails({
            users: accessGrants.users,
            groups: accessGrants.groups,
            accessKeys: accessGrants.accessKeys,
        });

        return permission.getRestDetails();
    };

    /**
     * Persists the state of the restriction as it is in the dialog.
     * If the action to be carried out is a create or update, returns the Rest representation of the restriction.
     * @param {String} checkboxId - the ID of the checkbox in the dialog corresponding to this restriction
     * @param existingRefId
     * @returns {RefRestriction} the modified restricted ref object or null if the object was not modified
     */
    BranchPermissionDialog.prototype.persistRestriction = function (checkboxId, existingRefId) {
        var isChecked = this.$('#' + checkboxId).prop('checked');

        if (!isChecked) {
            if (existingRefId) {
                return this.refRestrictions.get(existingRefId).remove();
            }

            return null;
        }

        return this.getRestPermissionFor(existingRefId, checkboxId);
    };

    /**
     * Validates the form, disables the submit button and saves all changed permissions.
     * If the save rest calls all succeed, calls the callback function and hides the dialog.
     * If one of the save rest calls fails, renders the returned error to the dialog.
     * @returns {promise|*}
     */
    BranchPermissionDialog.prototype.save = function () {
        this.showSpinner();
        if (!this.isValid()) {
            this.stopSpinner();

            return;
        }

        this.disableSubmit(true);

        var prevent = this._dialogOptions.prevent || {};
        // Count the existing permissions. Needed to figure out the type of change to this branch permission.
        // (Create, Update or Delete)
        var existingPermissions = _.compact(_.values(this._dialogOptions.prevent)).length;
        var requests = [
            this.persistRestriction('prevent-changes-checkbox', prevent.changes),
            this.persistRestriction('prevent-force-push-checkbox', prevent.force),
            this.persistRestriction('prevent-merge-checkbox', prevent.merge),
            this.persistRestriction('prevent-deletion-checkbox', prevent.deletion),
            this.persistRestriction('prevent-creation-checkbox', prevent.creation),
        ];

        // Remove any nulls - we only want to send any actually changed restriction types
        requests = _.compact(requests);

        // Partition the requests. They may contain removal promises.
        var requestPartition = _.partition(requests, function (obj) {
            // Test if the object is a promise
            return !!obj.then && typeof obj.then === 'function';
        });

        var promises = requestPartition[0];

        var operationType;

        if (requestPartition[1].length > 0) {
            var bulkAjax = ajax.rest({
                contentType: 'application/vnd.atl.bitbucket.bulk+json',
                url: nav
                    .rest('branch-permissions', 'latest')
                    [this.currentScopeNavMethod.nav](pageState[this.currentScopeNavMethod.state]())
                    .addPathComponents('restrictions')
                    .withParams({ avatarSize: 32 })
                    .build(),
                data: requestPartition[1],
                type: 'POST',
                statusCode: {
                    400: this.responseErrorHandler,
                    500: this.responseErrorHandler,
                },
            });

            promises.push(bulkAjax);

            if (!existingPermissions) {
                operationType = 'CREATE';
            }
        } else if (promises.length === existingPermissions) {
            // Deleting all existing permissions.
            operationType = 'DELETE';
        }

        if (!operationType) {
            operationType = 'UPDATE';
        }

        return $.when
            .apply($, promises)
            .always(this.stopSpinner)
            .then(this.mergeResponses)
            .done(
                function (response) {
                    return this.callback(response, operationType);
                }.bind(this)
            )
            .done(this.hide)
            .fail(this.renderErrors);
    };

    /**
     * Toggle the entity fields based on the state of the checkboxes.
     */
    BranchPermissionDialog.prototype.showHideEntityFields = function () {
        // disable prevent creation checkbox if type is BRANCH
        var matcherType = this._dialogOptions.editing
            ? this.getCurrentMatcher().type.id
            : this._dialog.$el.find('input[name="matcherType"]:checked').val();
        var preventCreation = this._dialog.$el.find('#prevent-creation-checkbox');

        if (matcherType === PICKER_TYPES.BRANCH) {
            preventCreation.prop({
                checked: false,
                disabled: true,
            });
        } else {
            preventCreation.prop('disabled', false);
        }

        var $checkBoxes = this._dialog.$el.find(
            '#restriction-types-checkbox-field input[type="checkbox"]'
        );
        var self = this;
        $checkBoxes.each(function () {
            var $val = $(this);
            var isShowing = $('#' + $val.attr('id') + '-entities').length > 0;
            var isChecked = $val.prop('checked');

            if (isChecked && !isShowing) {
                self.showEntityField($val);
            }

            if (!isChecked && isShowing) {
                self.hideEntityField($val);
            }
        });
    };

    BranchPermissionDialog.prototype.showEntityField = function ($val) {
        var entityField = bitbucketPluginRefRestriction.internal.feature.dialog.branchPermissionsDialog.entityField(
            {
                fieldFor: $val.attr('id'),
            }
        );

        var $entityField = $(entityField);
        $entityField.insertAfter($val.parent());
        this.initEntitySelector($entityField);
    };

    BranchPermissionDialog.prototype.hideEntityField = function ($val) {
        $('#' + $val.attr('id') + '-entities').remove();
    };

    BranchPermissionDialog.prototype.mergeResponses = function () {
        var mergedResponses = [];

        var addToMerged = function (item) {
            if (item instanceof Array) {
                item.forEach(addToMerged);
            } else if (item.type || item.matcher) {
                mergedResponses.push(item);
            }
        };
        _.forEach(arguments, addToMerged);

        return mergedResponses;
    };

    return BranchPermissionDialog;
});
