define('bitbucket-plugin-ref-restriction/internal/feature/table/branch-permissions-table', [
    '@atlassian/aui',
    '@atlassian/backbone-brace',
    'jquery',
    'lodash',
    'bitbucket/util/events',
    '../../model/ref-restriction-collection',
], function (AJS, Brace, $, _, events, RefRestrictionCollection) {
    'use strict';
    var BranchPermissionTableView = Brace.View.extend({
        namedAttributes: {
            collection: RefRestrictionCollection,
        },

        initialize: function (options) {
            this.options = options || {};
            _.bindAll(this, _.functions(this.constructor.prototype));
            this._scopeType = this.options.scopeType;
            this._projectSettingsUrl = this.options.projectSettingsUrl;
            this.collection = this.options.refRestrictions;
            this.listenTo(this.collection, 'add', this.addRow);
            this.listenTo(this.collection, 'remove', this.deleteRow);
            this.on('renderRow', this.fixRowHeights);
            events.on('window.resize.debounced', this.fixAllRowHeights);
            this.collection.each(function (refRestriction) {
                this.listenTo(refRestriction, 'change', this.editRow);
                this.addRow(refRestriction);
            }, this);
        },

        reRender: function (container) {
            var tableElement = $(
                bitbucketPluginRefRestriction.internal.feature.table.branchPermissionsTable.branchPermissionsTable(
                    {
                        scopeType: this._scopeType,
                    }
                )
            );
            container.html(tableElement);
            this.setElement($('.branch-permissions-table'));
            this.collection.each(function (refRestriction) {
                this.addRow(refRestriction);
            }, this);
        },

        destroy: function () {
            this.stopListening(this.collection);
            this.collection.each(function (refRestriction) {
                this.stopListening(refRestriction);
            }, this);
            $(this.el).removeData().unbind();
            this.off('renderRow');
            events.off('window.resize.debounced', this.fixAllRowHeights);
            this.remove();
            Brace.View.prototype.remove.call(this);
        },
    });

    BranchPermissionTableView.prototype.fixAllRowHeights = function () {
        var self = this;
        this.$el.find('tbody').each(function (i, el) {
            self.fixRowHeights($(el));
        });
    };

    /**
     * This is a hacky workaround for the way the ref restriction table currently works.
     * When a branch column cell is taller than a restriction row (e.g. a branch model restriction with only one
     * restriction), then the background colour of the entire row can look weird because of how table cells / tbody work.
     * i.e. the table cell that is taller won't cause the other table cells to match the height.
     * This will ensure that if the first column is taller, that the rest end up matching.
     *
     * @param {jQuery} $row
     */
    BranchPermissionTableView.prototype.fixRowHeights = function ($row) {
        // check if the first column is taller than the combined rest of the rows
        // if so, set the height of the sub rows.
        var $restRows = $row.find('tr:not(:first)');

        if ($restRows.length > 1) {
            // only a problem when there is one sub row
            return;
        }
        var firstRowHeight = $row.find('tr:first td').height();
        var restRowsHeight = $restRows.find('td:first').height();

        if (firstRowHeight > restRowsHeight) {
            $restRows.find('td').height(firstRowHeight);
        }
    };

    /**
     * Adds a row to the table view.
     * @param {RefRestriction} refRestriction
     */
    BranchPermissionTableView.prototype.addRow = function (refRestriction) {
        var matcher = refRestriction.getMatcher();
        var matcherType = matcher.getType();
        var refScopeType = refRestriction.getScope().type;

        var $parentRow = this.$el.find('tbody').filter(function (i, row) {
            var branchValue = row.dataset.branchValue;
            var branchType = row.dataset.branchType;
            var effectiveScopeType = row.dataset.effectiveScopeType;

            return (
                branchValue === matcher.getId() &&
                branchType === matcherType.getId() &&
                effectiveScopeType === refScopeType
            );
        });

        var $row;

        if (!$parentRow.length) {
            $row = $(
                bitbucketPluginRefRestriction.internal.feature.table.branchPermissionsTable.refRestrictionRow(
                    {
                        refRestriction: refRestriction.toJSON(),
                        projectSettingsUrl: this._projectSettingsUrl,
                        scopeType: this._scopeType,
                    }
                )
            ).appendTo(this.$el);
        } else {
            var $branchNameCell = $parentRow.find('.branch-name-cell');
            $branchNameCell.attr('rowspan', function incrementRowSpan(i, rowSpan) {
                return parseInt(rowSpan) + 1;
            });

            $row = $(
                bitbucketPluginRefRestriction.internal.feature.table.branchPermissionsTable.refRestrictionSubRow(
                    {
                        permission: refRestriction.toJSON(),
                    }
                )
            ).appendTo($parentRow);
        }

        refRestriction.on('change', this.editRow.bind(this));
        this.renderRowDetails($row, refRestriction);
        _.defer(this.trigger.bind(this, 'renderRow', $row));
    };

    /**
     * Updates a the table row containing the refRestriction
     * @param {RefRestriction} refRestriction
     * @this {BranchPermissionTableView}
     */
    BranchPermissionTableView.prototype.editRow = function (refRestriction) {
        var $rowToReplace = this.$el.find('tr[data-id="' + refRestriction.getId() + '"]');
        var $row = $(
            bitbucketPluginRefRestriction.internal.feature.table.branchPermissionsTable.refRestrictionSubRow(
                {
                    permission: refRestriction.toJSON(),
                }
            )
        ).replaceAll($rowToReplace);
        this.renderRowDetails($row, refRestriction);
        this.trigger('renderRow', $row);
    };

    /**
     * Deletes the table row containing the refRestriction
     * @param {RefRestriction} refRestriction
     */
    BranchPermissionTableView.prototype.deleteRow = function (refRestriction) {
        this.stopListening(refRestriction);
        var $row = this.$el.find('tr[data-id="' + refRestriction.getId() + '"]');
        var $parent = $row.parent('tbody');
        $row.remove();

        if ($parent.children('tr[data-id]').length === 0) {
            $parent.remove();
        } else {
            $parent
                .find('.branch-name-cell')
                .attr('rowspan', function decrementRowSpan(i, rowSpan) {
                    return parseInt(rowSpan) - 1;
                });
        }
    };

    /**
     * Replaces the row's users and groups cell with the users and groups contained in details
     * @param {jQuery} $row
     * @param {RefRestriction} refRestriction
     */
    BranchPermissionTableView.prototype.renderRowDetails = function ($row, refRestriction) {
        var $exemptionCell = $row.find('.cell-exemptions');
        var users = refRestriction.getUsers() || [];
        var groups = refRestriction.getGroups() || [];
        var accessKeys = refRestriction.getAccessKeys() || [];
        $exemptionCell.html(this.getRenderedExemptions(users, groups, accessKeys));
        $row.find('.deleted-user').tooltip({
            title: function () {
                return AJS.I18n.getText('bitbucket.web.branch.permission.deleted.user.tooltip');
            },
            gravity: 'e',
        });

        $row.find('.aui-lozenge[title]').tooltip();
    };

    /**
     * Calls a soy template to get the HTML for the specified exemptions
     * @param {Array<Object>} users
     * @param {Array<string>} groups
     * @param {Array<Object>} accessKeys
     * @returns {HTMLElement}
     */
    BranchPermissionTableView.prototype.getRenderedExemptions = function (
        users,
        groups,
        accessKeys
    ) {
        return bitbucketPluginRefRestriction.internal.feature.table.branchPermissionsTable.exemptionsList(
            {
                users: users,
                groups: groups,
                accessKeys: accessKeys,
            }
        );
    };

    /**
     * Shows a spinner on the specified row
     * @param {jQuery} $row
     */
    BranchPermissionTableView.showRowSpinner = function ($row) {
        $row.find('.spinner').addClass('show').spin();
    };

    /**
     * Hides a spinner on the specified row
     * @param {jQuery} $row
     */
    BranchPermissionTableView.hideRowSpinner = function ($row) {
        $row.find('.spinner').removeClass('show').spinStop();
    };

    /**
     * Show the delete button action on the specified row
     * @param {jQuery} $row
     */
    BranchPermissionTableView.showDeleteButton = function ($row) {
        $row.find('.remove-ref-restriction-button').show();
    };

    /**
     * Hide the delete button action on the specified row
     * @param {jQuery} $row
     */
    BranchPermissionTableView.hideDeleteButton = function ($row) {
        $row.find('.remove-ref-restriction-button').hide();
    };

    return BranchPermissionTableView;
});
