(function (syncing) {
    /**
     * Registers a new chaperone tour
     *
     * @param {Object} opts - The tour definition and options
     * ... {String} id - unique key that will be used on the server to persist which steps have been dismissed
     * ... {Array[Object]} steps - array of steps in the tour
     * @param {Object} deps - the library dependencies
     * ... {Object} brace - backbone brace library reference
     * ... {Object} jQuery - jQuery brace library reference
     * ... {Object} _ - underscore library reference
     */
    // eslint-disable-next-line no-undef
    Chaperone.register = function (opts, deps) {
        opts = opts || {};
        var _jQuery = deps.jQuery || window.jQuery;

        var defaultLibs = {
            underscore: window._,
            brace: window.Brace,
        };

        var defaultOpts = {
            numbered: true,
        };

        deps = _jQuery.extend({}, defaultLibs, deps);
        opts = _jQuery.extend({}, defaultOpts, opts);

        var __ = deps.underscore;
        var _Brace = deps.brace;

        /**
         * Model representing a single step in tour
         *
         * @param {String} num - Number displayed in callout
         * @param {String, Function} selector - css selector of element the callout will be positioned against
         * @param {String, Function} dismissingSelectors - css selectors of elements that when clicked will dismiss the step.
         * @param {String} dir - The direction of the callout faces - e,w,s,n
         * @param {String} text - text for the callout, will be html escaped
         * @param {String} html - html for the callout, will NOT be html escaped
         * @param {String} dismissed - has the step been followed by the user
         * @param {Function} onDismiss - function called when the step been followed by the user
         * @param {Function} onShow - function called when the callout hs been displayed
         * @param {String} styleClass - class that will be added to pin so additional styling can be applied
         */
        var StepModel = _Brace.Model.extend({
            defaults: {
                width: 250,
            },
            namedAttributes: {
                num: Number,
                selector: undefined,
                dismissingSelectors: undefined,
                dir: String,
                text: String,
                html: String,
                styleClass: String,
                dismissed: Boolean,
                showNumber: Boolean,
                onDismiss: Function,
                onShow: Function,
                width: Number,
                minHeight: Number,
            },
            /**
             * Resolves the css selector string for target. The target being the element the callout will be positioned against.
             * @returns {String} selector
             */
            resolveSelector: function () {
                var selector = this.getSelector();

                if (__.isFunction(selector)) {
                    selector = selector();
                }

                return selector;
            },
            /**
             * Resolves the css selector string for dismissing elements. The elements that when clicked will dismiss the step.
             * @returns {String} selector
             */
            resolveDismissingSelectors: function () {
                var selector = this.getDismissingSelectors();

                if (__.isFunction(selector)) {
                    selector = selector();
                }

                return selector;
            },
        });

        /**
         * Representing a single step in tour
         *
         * @see StepModel for inherited attributes
         * @param {ChildStepCollection} steps -
         */
        var ChildStepModel = StepModel.extend({
            namedAttributes: {
                required: Boolean,
            },
            defaults: {
                required: true,
                width: 250,
            },
        });

        /**
         * A collection of child steps, represents a tour that stems off each primary step.
         * A primary step is only completed after all its required child steps have been dismissed.
         */
        var ChildStepCollection = _Brace.Collection.extend({
            model: ChildStepModel,
        });

        /**
         * Representing a single step in tour. A primary step can have child steps, which all need to be dismissed
         * before it is marked as completed.
         *
         * @see StepModel for inherited attributes
         * @param {ChildStepCollection} steps
         */
        var PrimaryStepModel = StepModel.extend({
            initialize: function () {
                // if we have child steps bubble the dismissed event up
                var subSteps = this.getSteps();

                if (subSteps) {
                    this.listenTo(subSteps, 'change:dismissed', function () {
                        this.trigger('childStepDismissed');
                    });
                }
            },
            namedAttributes: {
                steps: ChildStepCollection,
            },
        });

        /**
         *  A collection of steps that make up the tour.
         */
        var PrimaryStepCollection = _Brace.Collection.extend({
            model: PrimaryStepModel,

            /**
             * Gets all the ids of primary steps that have been completed. A step is completed if:
             * - All the required child steps have been dismissed.
             * - If no child steps, if primary step has been dismissed.
             * @returns Array[String]
             */
            getCompleted: function () {
                var dismissed = this.filter(function (model) {
                    if (model.getSteps()) {
                        var requiredSteps = model.getSteps().filter(function (step) {
                            return step.getRequired();
                        });

                        if (requiredSteps.length) {
                            return requiredSteps.pop().getDismissed();
                        } else {
                            return model.getDismissed();
                        }
                    } else {
                        return model.getDismissed();
                    }
                });

                return __.pluck(dismissed, 'id');
            },
        });

        /**
         * A mixin for views that position the element relative to target & the direction it should face.
         */
        var PosMixin = {
            /**
             * Positions the element relative to target & the direction it should face.
             * @param {String} dir - direction pin or callout should face
             * @param {jQuery} $target - relative element
             */
            pos: function (dir, $target) {
                dir = dir || 's';
                this['pos' + dir.toUpperCase()]($target);
            },

            /**
             * Positions element facing east relative to target
             * @param {jQuery} $target
             */
            posE: function ($target) {
                var offset = $target.offset();
                offset.top = offset.top + $target.outerHeight() / 2;
                offset.left = offset.left - this.$el.outerWidth() - 10;
                offset.marginTop = -this.$el.outerHeight() / 2;
                this.$el.css(offset).addClass('chaperone-dir-e');
            },
            /**
             * Positions element facing south relative to target
             * @param {jQuery} $target
             */
            posS: function ($target) {
                var offset = $target.offset();
                offset.top = offset.top + $target.outerHeight() + 10;
                this.$el.css(offset).addClass('chaperone-dir-s');
            },
            /**
             * Positions element facing north relative to target
             * @param {jQuery} $target
             */
            posN: function ($target) {
                var offset = $target.offset();
                offset.top = offset.top - this.$el.outerHeight() - 10;
                this.$el.css(offset).addClass('chaperone-dir-n');
            },
            /**
             * Positions element facing west relative to target
             * @param {jQuery} $target
             */
            posW: function ($target) {
                var offset = $target.offset();
                offset.top = offset.top + $target.outerHeight() / 2;
                offset.left = offset.left + $target.outerWidth() + 10;
                offset.marginTop = -this.$el.outerHeight() / 2;
                this.$el.css(offset).addClass('chaperone-dir-w');
            },
        };

        /**
         * View that represents a single step in the tour
         */
        var StepView = _Brace.View.extend({
            mixins: [PosMixin],
            className: 'chaperone',

            /**
             * Listens for click on target element to mark the step as dismissed.
             * @param {jQuery} $target the element for the callout to be rendered relative to.
             */
            enhanceTarget: function ($target) {
                if (!$target.hasClass('chaperone')) {
                    $target.addClass('chaperone-target').attr('data-chaperone-id', this.model.id);
                    if (!this.model.resolveDismissingSelectors()) {
                        // If no dismissing selectors, make the main selector dismiss
                        $target.one(
                            'click',
                            __.bind(function (e) {
                                this.dismiss();
                                // Let event bubble before we remove class
                                __.defer(function () {
                                    $target.removeClass('chaperone-target');
                                });
                            }, this)
                        );
                    }
                }
            },
            applyDismissers: function () {
                var dismissingSelectors = this.model.resolveDismissingSelectors();

                if (dismissingSelectors) {
                    _jQuery(dismissingSelectors).one('click', __.bind(this.dismiss, this));
                }
            },
            dismiss: function () {
                if (!this.model.getDismissed()) {
                    this.model.setDismissed(true);
                    this.$el.remove();
                    var dismissCallback = this.model.getOnDismiss();

                    if (dismissCallback) {
                        dismissCallback();
                    }
                }
            },

            updatePos: function () {
                var selector = this.model.resolveSelector();
                var $target = _jQuery(selector);

                if ($target) {
                    this.pos(this.model.getDir(), $target);
                    // the dom node may have been replaced, e.g ajax refresh of content, so enhance and apply dismissers again
                    this.enhanceTarget($target);
                    this.applyDismissers();
                }
            },
            getTarget: function () {
                var selector = this.model.resolveSelector();
                var $target = _jQuery(selector);

                return $target.size() ? $target : null;
            },
            /**
             * Renders and positions callout relative to target.
             * @param {jQuery} $target the element for teh callout to be rendered relative to.
             */
            renderStep: function () {
                var additionalClasses = this.model.getStyleClass();

                if (additionalClasses) {
                    this.$el.addClass(additionalClasses);
                }
                if (this.model.getShowNumber()) {
                    this.$el.addClass('chaperone-numbered');
                }
                if (this.model.getMinHeight()) {
                    this.$el.css('min-height', this.model.getMinHeight());
                }
                this.$el
                    .css({
                        width: this.model.getWidth(),
                    })
                    // eslint-disable-next-line no-undef
                    .html(_jQuery(chaperone.Templates.pin(this.model.toJSON())))
                    .appendTo('body');
            },
            /**
             * Renders step if the target is found in dom
             */
            render: function () {
                var $target = this.getTarget();

                if ($target) {
                    this.renderStep();
                    this.updatePos();
                    this.applyDismissers();
                    var onShow = this.model.getOnShow();

                    if (onShow) {
                        onShow();
                    }
                }
            },
        });

        /**
         * The Chaperone module is responsible for
         * - Creating all the views for the tour steps
         * - Syncing which steps have been completed to the server
         * - Rendering the correct 'next' step
         */
        var ChaperoneModule = _Brace.View.extend({
            initialize: function (options) {
                __.bindAll(this, 'renderNextStep');
                this.options = options;
                // go to server and mark completed steps before we create collection
                this.syncAndUpdateStepDefinitions(options).done(
                    __.bind(function (items) {
                        // Create backbone collection representing to hold all the step definitions
                        this.createStepCollection(items, options);
                        // Create views for all the step definitions
                        this.createStepViews();
                        // Apply listeners for when we need to recalculate which step to show.
                        this.applyUpdateTriggers();
                        // Apply listeners for when we need to recalculate the position of the current step.
                        this.applyRepositionTriggers();
                        // Finally render next valid step (if there is one).
                        this.renderNextStep();
                    }, this)
                );
            },

            applyRepositionTriggers: function () {
                _jQuery(window).resize(__.bind(this.repositionStep, this));
            },

            applyUpdateTriggers: function () {
                // Provide a way for developers to trigger a refresh of pins.
                _jQuery(document).bind('update.chaperone', this.renderNextStep);
                // Whenever an ajax request completes, more content could have been added to the dom, so we need
                // to revaluate which step to show.
                _jQuery(document).ajaxStop(
                    __.bind(function () {
                        __.defer(this.renderNextStep);
                    }, this)
                );
                // Whenever we change the url update steps
                _jQuery(window).bind(
                    'popstate hashchange',
                    __.bind(function () {
                        __.defer(this.renderNextStep);
                    }, this)
                );
            },

            /**
             * Creates the collection of primaruy steps
             * @param Array[Object] stepDefinitions
             */
            createStepCollection: function (stepDefinitions) {
                this.stepCollection = new PrimaryStepCollection(stepDefinitions);
                this.listenTo(
                    this.stepCollection,
                    'change:dismissed childStepDismissed',
                    this.renderNextStep
                );
                if (this.options.sync !== false) {
                    this.applySyncing();
                }
            },
            /**
             * Adds the server side persistence to the step collection
             */
            applySyncing: function () {
                this.listenTo(
                    this.stepCollection,
                    'change:dismissed childStepDismissed',
                    function () {
                        syncing.syncToServer(
                            this.options.id,
                            JSON.stringify(this.stepCollection.getCompleted())
                        );
                    }
                );
            },

            /**
             * Repositions the current step relative to its target.
             */
            repositionStep: function () {
                if (this.currentStep) {
                    this.currentStep.updatePos();
                }
            },

            /**
             * Renders steps to screen.
             */
            createStepViews: function () {
                this.stepViews = [];
                this.stepCollection.each(
                    __.bind(function (model) {
                        var stepView = new StepView({ model: model });
                        this.stepViews.push(stepView);
                        var subSteps = model.getSteps();

                        if (subSteps) {
                            subSteps.each(function (model) {
                                var stepView = new StepView({ model: model });
                                this.stepViews.push(stepView);
                            }, this);
                        }
                    }, this)
                );
            },
            /**
             * Renders steps to screen
             */
            renderNextStep: function () {
                var potentialStep = __.find(this.stepViews, function (view) {
                    return !view.model.getDismissed();
                });

                if (potentialStep) {
                    var $el = _jQuery(potentialStep.model.resolveSelector());

                    if ($el.size() && $el.is(':visible')) {
                        if (this.currentStep != potentialStep) {
                            // avoid double rendering
                            this.currentStep = potentialStep;
                            this.currentStep.render();
                        } else {
                            this.currentStep.updatePos();
                        }
                    } else {
                        potentialStep.$el.remove();
                        this.currentStep = null;
                    }
                }
                var stepsToRemove = __.without(this.stepViews, potentialStep);
                __.each(stepsToRemove, function (step) {
                    step.$el.remove();
                });
            },

            dismissTour: function () {
                __.each(this.stepViews, function (step) {
                    step.model.setOnDismiss(function () {});
                    step.dismiss();
                });
            },

            /**
             * 1. Adds num property based on index, this is the number of the step shown in callout
             * 2. Marks which steps have already been completed
             * 3. Maps selector property for id if id has not been defined
             */
            _updateStepDefinitions: function (options, completedSteps) {
                var steps = [];
                __.each(options.steps, function (primaryStep, idx) {
                    primaryStep.num = idx + 1;
                    primaryStep.showNumber = options.numbered;
                    primaryStep.id = primaryStep.id || primaryStep.selector;
                    primaryStep.dismissed = __.contains(completedSteps, primaryStep.id);
                    if (primaryStep.steps) {
                        primaryStep.steps = __.map(primaryStep.steps, function (step) {
                            step.id = step.id || step.selector;
                            step.dismissed = primaryStep.dismissed;

                            return step;
                        });
                    }
                    steps.push(primaryStep);
                });

                return steps;
            },

            /**
             * If syncing is enabled goes to server to get already completed steps to mark the step definitions as
             * completed.
             */
            syncAndUpdateStepDefinitions: function (options) {
                var deferred = _jQuery.Deferred();

                if (options.sync != false) {
                    syncing.syncFromServer(options.id, _jQuery).done(
                        __.bind(function (completedSteps) {
                            deferred.resolve(this._updateStepDefinitions(options, completedSteps));
                            this.trigger('syncComplete', completedSteps);
                        }, this)
                    );
                } else {
                    deferred.resolve(this._updateStepDefinitions(options));
                }

                return deferred;
            },
        });

        return new ChaperoneModule(opts);
    };
    // eslint-disable-next-line no-undef
})(Chaperone._internal.syncing);
