/**
 * @fileoverview
 * We use a number of events to communicate with the JavaScript provided by the host application to allow
 * us to enable and disable keyboard shortcuts.
 *
 * initialize.keyboardshortcuts - The host application should trigger this event when the page is ready
 * to bind the keyboard shortcuts. The host application must ensure that any necessary internationalization
 * resources are available before triggering this event.
 *
 * register-contexts.keyboardshortcuts - keycommands.js will trigger this event and pass the shortcutsRegistry
 * object to the bound function. The host application should bind a function to this event which will take the
 * shortcutsRegistry object and call enableContext: for each context fow which it wishes to enable shortcuts e.g.
 *
 * AJS.bind("register-contexts.keyboardshortcuts", function (e, data) {
 *      data.shortcutRegistry.enableContext("viewcontent");
 * });
 *
 * remove-bindings.keyboardshortcuts - The host application can trigger this event to disable all keyboard
 * shortcuts.
 *
 * add-bindings.keyboardshortcuts - The host application can trigger this event to re-enable all keyboard
 * shortcuts if 'remove-bindings.keyboardshortcuts' was triggered previously.
 *
 *
 *
 * Example product code:
 *
 * AJS.bind("register-contexts.keyboardshortcuts", function(e, data) {
 *     data.shortcutRegistry.enableContext("global");
 *     data.shortcutRegistry.enableContext("my-context");
 * });
 *
 * AJS.trigger("initialize.keyboardshortcuts");
 *
 */
define('atlassian/shortcuts/js/keycommands', [
    'jquery',
    'atlassian/shortcuts/js/when-i-type',
], function ($, whenIType) {
    function convertToWhenIType(shortcuts, context) {
        var isMac = navigator.platform.indexOf('Mac') !== -1;

        return shortcuts
            .filter(function (shortcut) {
                return shortcut.context === context;
            })
            .map(function (item) {
                var operation = item.op;
                var param = item.param;
                var params;
                if (operation === 'execute' || operation === 'evaluate') {
                    params = [item['function']];
                } else if (/^\[[^\]\[]*,[^\]\[]*\]$/.test(param)) {
                    // pass in an array to send multiple params
                    try {
                        params = JSON.parse(param);
                    } catch (e) {
                        console.error(
                            'When using a parameter array, array must be in strict JSON format: ' +
                                param
                        );
                    }

                    if (!$.isArray(params)) {
                        console.error(
                            'Badly formatted shortcut parameter. String or JSON Array of parameters required: ' +
                                param
                        );
                    }
                } else {
                    params = [param];
                }

                return {
                    operation: operation,
                    keys: item.keys,
                    params: params,
                };
            })
            .flatMap(function (item) {
                var operation = item.operation;
                var params = item.params;
                return item.keys.map(function (shortcutList) {
                    if (isMac) {
                        shortcutList = shortcutList.map(function (shortcutString) {
                            return shortcutString.replace(/ctrl/i, 'meta');
                        });
                    }

                    var newShortcut = whenIType(shortcutList);
                    newShortcut[operation].apply(newShortcut, params);
                    return newShortcut;
                });
            });
    }

    AJS.bind('initialize.keyboardshortcuts', function () {
        var data = keyboard_shortcuts();
        var shortcuts = data && data.shortcuts;

        if (!shortcuts) throw new Error('Server returned no shortcuts.');
        AJS.trigger('shortcuts-loaded.keyboardshortcuts', {
            shortcuts: shortcuts,
        });

        var enabledShortcuts = [];

        /**
         * The shortcutRegistry object allows the host application to enable shortcuts for a given context.
         */
        var shortcutRegistry = {
            /**
             * @param {String} context. All shortcuts with the given context will be enabled.
             */
            enableContext: function (context) {
                var contextShortcuts = convertToWhenIType(shortcuts, context);
                Array.prototype.push.apply(enabledShortcuts, contextShortcuts);
            },
        };

        var addKeyBindings = function () {
            AJS.trigger('register-contexts.keyboardshortcuts', {
                shortcutRegistry: shortcutRegistry,
            });
        };
        addKeyBindings();
        AJS.bind('add-bindings.keyboardshortcuts', addKeyBindings);

        var removeKeyBindings = function () {
            $.each(enabledShortcuts, function () {
                this.unbind();
            });
            enabledShortcuts = [];
        };
        AJS.bind('remove-bindings.keyboardshortcuts', removeKeyBindings);
    });
});

require('atlassian/shortcuts/js/keycommands');
