define('bitbucket-plugin-search/internal/feature/search/internal/search', ['exports', 'jquery', 'lodash', 'react', 'react-dom', 'bitbucket-plugin-search/internal/component/search-urls', 'bitbucket/internal/impl/request', '../../search-common/search-entities', '../../search-common/search-request', './analytics', './search-data-provider', './search-repositories', './search-results-table'], function (exports, _jquery, _lodash, _react, _reactDom, _searchUrls, _request, _searchEntities, _searchRequest, _analytics, _searchDataProvider, _searchRepositories, _searchResultsTable) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.search = search;
    exports.createSearch = createSearch;

    var _jquery2 = _interopRequireDefault(_jquery);

    var _lodash2 = _interopRequireDefault(_lodash);

    var _react2 = _interopRequireDefault(_react);

    var _reactDom2 = _interopRequireDefault(_reactDom);

    var _searchUrls2 = _interopRequireDefault(_searchUrls);

    var _request2 = _interopRequireDefault(_request);

    var _searchEntities2 = _interopRequireDefault(_searchEntities);

    var _analytics2 = _interopRequireDefault(_analytics);

    var _searchDataProvider2 = _interopRequireDefault(_searchDataProvider);

    var _searchRepositories2 = _interopRequireDefault(_searchRepositories);

    var _searchResultsTable2 = _interopRequireDefault(_searchResultsTable);

    function _interopRequireDefault(obj) {
        return obj && obj.__esModule ? obj : {
            default: obj
        };
    }

    var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) {
        return typeof obj;
    } : function (obj) {
        return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
    };

    var STATUS_OK = 200;
    var STATUS_SERVER_ERROR = 500;

    // List of all entity types currently supported for rendering by search as a _primary_ result (repo search is secondary)
    var SUPPORTED_ENTITIES = [_searchEntities2.default.CODE];

    var REPOSITORY_SCOPE = /repo\:\S/;

    /**
     * Given a SearchRequest object, perform a search.
     * @param {Object} searchRequest - The parameters for this search.
     * @returns {Promise} request promise
     */
    function search(searchRequest) {
        var start = Date.now();

        var ajaxPromise = _request2.default.rest({
            type: 'POST',
            url: _searchUrls2.default.searchRestUrl(),
            data: searchRequest,
            statusCode: {
                400: false, // 400 is returned for QueryInvalidException,
                500: false // 500 is also handled gracefully by the search UI
            }
        });

        ajaxPromise.then(function (data) {
            var time = Date.now() - start;
            var repository = data.scope.repository;
            var project = repository ? repository.project : data.scope.project;
            _analytics2.default.resultsLoaded({
                repository: repository,
                project: project,
                time: time,
                query: searchRequest.query,
                results: data
            });
        });

        return ajaxPromise;
    }

    /**
     * Sets up and creates the search result UI within an element, with an initial query.
     * @param {jQuery|HTMLElement} el Element to render results in
     * @param {string} query - The initial query string
     * @param {Object} limits - The search paging limits
     * @param {Number} limits.primary - Primary search page size (eg. main results)
     * @param {Number} limits.secondary - Secondary search page size (eg. sidebar results)
     * @returns {Promise} Promise that resolves when the initial search completes
     */
    function createSearch(el, query, limits) {
        var searchResultsTable = void 0;
        var loadedResults = 0;

        // setup search data provider for query
        var $el = (0, _jquery2.default)(el);
        var $primarySearchPlaceholder = $el.find('.primary-results-container');
        var $secondarySearchPlaceholder = $el.find('.secondary-results-container');

        $primarySearchPlaceholder.empty(); // also unbinds event handlers

        // show the 'searching' spinner while we do the initial search
        $primarySearchPlaceholder.html(bitbucketPluginSearch.internal.feature.search.search.searching());
        var $searching = $el.find('.code-search-searching');
        $searching.find('.spin-container').spin();
        $searching.show();

        _reactDom2.default.render(_react2.default.createElement(_searchRepositories2.default, {
            query: query,
            limit: limits.primary,
            active: !REPOSITORY_SCOPE.test(query)
        }), $secondarySearchPlaceholder.get(0));

        function handleServerError() {
            var $results = $primarySearchPlaceholder.find('.primary-results');
            var errorHtml = bitbucketPluginSearch.internal.feature.search.search.serverError();

            if (searchResultsTable) {
                searchResultsTable.suspend();
            }

            if ($results.length) {
                $results.append(errorHtml);
            } else {
                $primarySearchPlaceholder.html(errorHtml);
            }
        }

        var handleSearchResult = function handleSearchResult(searchResult) {
            // setup SearchResultsTable for primary results
            var primaryResultType = _lodash2.default.findKey(searchResult, { category: 'primary' });
            var primaryResult = searchResult[primaryResultType];
            var dataProvider = new _searchDataProvider2.default({
                query: query,
                search: search,
                type: primaryResultType,
                limits: limits
            }, searchResult);

            dataProvider.on('data-request-failed', function (error) {
                handleServerError(error);
            });

            var repository = searchResult.scope.repository;
            var project = repository ? repository.project : searchResult.scope.project;
            var scope = {
                repository: repository,
                project: project
            };
            loadedResults = primaryResult.values.length;

            // TODO handle this better
            $primarySearchPlaceholder.html(bitbucketPluginSearch.internal.feature.search.search.codeResults({
                totalMatches: primaryResult.count,
                totalResults: primaryResult.count,
                loadedResults: loadedResults,
                scope: scope
            }));

            $el.on('click', '.learnmore', function () {
                return _analytics2.default.learnMoreClicked();
            });

            searchResultsTable = new _searchResultsTable2.default($el.find('.primary-results'), {
                dataProvider: dataProvider,
                scope: scope,
                querySubstituted: _lodash2.default.get(searchResult, 'query.substituted', false)
            });
            searchResultsTable.init();

            var $progress = $el.find('.result-load-progress');
            dataProvider.on('data-loaded', function (data) {
                loadedResults += data.length;
                if (dataProvider.reachedEnd) {
                    $progress.hide();
                } else {
                    $progress.html(bitbucketPluginSearch.internal.feature.search.search.resultLoadProgress({
                        loaded: loadedResults,
                        total: primaryResult.count
                    }));
                }
            });

            return searchResultsTable;
        };

        // perform initial search
        return search((0, _searchRequest.searchFor)(query, SUPPORTED_ENTITIES, limits)).then(handleSearchResult).fail(function (xhr, textError, errorThrown, data) {
            if (xhr.status === STATUS_OK) {
                // this _should_ only happen if the user's authentication has changed in BbS. In this case they'll get the
                // "your login changed" dialog which allows them to reload the page, but that still fails the promise. We
                // want to actually display the query result anyway, since the user will be notified it could be out of date.
                return handleSearchResult(data);
            } else if (xhr.status < STATUS_SERVER_ERROR) {
                $primarySearchPlaceholder.html(bitbucketPluginSearch.internal.feature.search.search.badQuery({
                    query: query,
                    reason: (typeof data === 'undefined' ? 'undefined' : _typeof(data)) === 'object' ? _lodash2.default.get(data, 'errors.0.message') : textError
                }));
            } else {
                handleServerError(xhr.responseJSON);
            }
        }).always(function () {
            $searching.hide();
        });
    }
});