define('jira-integration-plugin/jira-create-issue-form', [
    'jquery',
    'jira-integration-plugin/lodash',
    'jira-integration-plugin/jira-create-issue-form-data',
    'jira-integration-plugin/jira-create-issue-form-field-helper',
    'jira-integration-plugin/fields',
], function ($, _, JiraCreateIssueFormData, JiraCreateIssueFormFieldHelper, JiraIntegrationFields) {
    var templates = window.jiraIntegration.templates;

    JiraCreateIssueForm.defaults = {
        allowUnsupportedFields: false,
        ignoreFieldsWithDefaultValue: true, // Filter out fields that has default value ~ to make the form shorter
        excludedField: ['project', 'issuetype', 'reporter'], // don't need display Reporter, use user config in applink as default reporter.
        formClass: 'jira-interation-create-issue-form',
        requiredFieldsOnly: true,
        renderSummaryAndDescription: false,
        get$unsupportedFieldMessage: function (unSupportedFields, createIssueUrl) {
            return $(templates.jiraCreateIssueForm.unsupportedFieldsMessage({
                unsupportedFields: unSupportedFields,
                createIssueUrl: createIssueUrl,
            }));
        },
        get$unexpectedError: function (message) {
            return $(aui.message.error({content: AJS.escapeHtml(message)}));
        },
        get$communicationError: function (message) {
            return $(templates.jiraCreateIssueForm.communicationErrorMessage({
                message: message,
            }));
        },
        get$unauthorizeMessage: function (applicationName) {
            return $(templates.jiraCreateIssueForm.authorize({
                extraClasses: 'jira-oauth-dialog',
                applicationName: applicationName,
            }));
        },
        get$unsupportedServerMessage: function (serverUrl) {
            return $(templates.jiraCreateIssueForm.unsupportedServerMessage({
                serverUrl: serverUrl,
            }));
        },
        get$unrenderableRequiredFieldsMessage: function (fields, serverUrl) {
            var names = fields.map(function (field) {
                return AJS.escapeHtml(field.name);
            }).join(', ');
            return $(templates.jiraCreateIssueForm.unrenderableRequiredFieldsMessage({
                names: names,
                count: fields.length,
                serverUrl: serverUrl,
            }));
        },
        ajax: function (options) {
            alert('JiraCreateIssueForm requires the option \'ajax(options)\' to be specified and respond like jQuery.ajax.\n' +
                'This method should handle response status codes other than 200 and 500 (those are handled by us).');
        },
    };

    function JiraCreateIssueForm (opts) {
        this.configuration = _.extend({}, JiraCreateIssueForm.defaults, opts);

        // if renderSummaryAndDescription is true, definitely excludedField should include summary and description
        if (this.configuration.renderSummaryAndDescription) {
            this.configuration.excludedField.push('summary', 'description');
        }

        this._init();
    }

    JiraCreateIssueForm.prototype._trigger = function (eventName, options) {
        var eventHandler = this.configuration[eventName];
        if (eventHandler) {
            eventHandler.apply(this, Array.prototype.slice.call(arguments, 1));
        }
    };

    JiraCreateIssueForm.prototype._selectServer = function (serverId, force) {
        var me = this;

        // prevent multiple requests for issueCreation supported if the selected server was already
        // selected
        if (this.currentServerId === serverId && !force) {
            return;
        }

        this.currentServerId = serverId;
        this.formIsLoading(this.defaultFields.server);
        JiraCreateIssueFormData.isIssueCreationSupported(serverId, this.configuration.ajax)
            .done(function (issueCreationSupported) {
                if (issueCreationSupported) {
                    me._loadProjectData(serverId);
                    me.formLoadingCompleted(me.defaultFields.server);
                } else {
                    var displayUrl = JiraCreateIssueFormData.getServerById(serverId).displayUrl;
                    me._handleUnsupportedServer(displayUrl);
                }
            })
            .fail(function (error) {
                if (error[0] && error[0].authenticationUri) {
                    me._handleAuthorizeError(error[0]);
                } else {
                    me._handleAjaxError(error);
                }
            });
    };

    JiraCreateIssueForm.prototype._bindEventListener = function () {
        var me = this;
        this.defaultFields.server.on('change', function (e) {
            me.fieldValues = _.extend({}, me.fieldValues, me._getFieldValues());
            me.resetForm(true);
            var serverId = this.value;
            if (serverId) {
                me._selectServer(serverId);
            } else {
                JiraCreateIssueFormFieldHelper.resetSelectField(me.defaultFields.project);
                me.defaultFields.project.trigger('change');
            }
            me._trigger('onServerChanged', this.value);
        });
        this.defaultFields.project.on('change', function () {
            var projectId = this.value || $(this).select2('data').value;
            var issueType = $(me.defaultFields.issueType).select2('data');
            me.fieldValues = _.extend({}, me.fieldValues, me._getFieldValues());
            if (projectId) {
                $('option[value=""]', me.defaultFields.project).remove(); // We don't need to show "select a project" any more
                me._loadIssueType(me.currentServerId, projectId, issueType);
            } else {
                JiraCreateIssueFormFieldHelper.resetSelectField(me.defaultFields.issueType);
            }
            me._trigger('onProjectChanged', this.value);
        });
        this.defaultFields.issueType.on('change', function () {
            me.fieldValues = _.extend({}, me.fieldValues, me._getFieldValues());
            me.resetForm(true);
            if (this.value) {
                me._loadFields(me.fieldValues);
            } else {
                me.$containerRequireField && me.$containerRequireField.html('');
                removeAuiMessages(this.$form);
            }
            me._trigger('onTypeChanged', this.value);
        });
    };

    JiraCreateIssueForm.prototype._getFieldValues = function () {
        return this._getJSON({
            getFieldJSON: JiraIntegrationFields.getInternalJSON,
            $form: this.$form,
        });
    };

    JiraCreateIssueForm.prototype._init = function () {
        this.container = $(this.configuration.container);
        this.fieldValues = {};

        if (this.container.length) {
            this._renderForm();
            this._bindEventListener();
            this._loadServerData();
        } else {
            throw new Error('container property should be defined.');
        }
    };

    JiraCreateIssueForm.prototype._renderForm = function () {
        var _this = this;
        this.$form = $(templates.jiraCreateIssueForm.form({
            formClass: this.configuration.formClass,
        })).on('aui-valid-submit', function (e) {
            if (_this.configuration.onSubmit) {
                e.preventDefault();
                _this.configuration.onSubmit();
            }
        }).on('focus', '.fake-tabbable', function () {
            // fake-tabbable is a hack that prevents the focus being moved away from the form when the first select
            // element is opened; this makes sure the focus is on the input and not on fake-tabbable when we focus on the form
            // see https://ecosystem.atlassian.net/browse/AUI-4403
            const selectDropEl = $('div#select2-drop.select2-drop-active').children()[0];
            if (selectDropEl && selectDropEl.children.length) {
                selectDropEl.getElementsByTagName('input')[0].focus();
            }
        });

        if (this.configuration.renderSummaryAndDescription) {
            this._renderSummaryAndDescription();
        }

        this.container.append(this.$form);
        this.defaultFields = this._getDefaultFields();

        // Enrich default fields style to select2
        _.each(this.defaultFields, _.bind(function (field) {
            this.createSelect2WithIcon($(field));
        }, this));

        this.$containerRequireField = $('.create-issue-required-fields', this.container);
        this._trigger('onFormRendered');
    };

    JiraCreateIssueForm.prototype._renderSummaryAndDescription = function () {
        var defaulFieldsContainer = $('.create-issue-default-fields', this.$form);

        defaulFieldsContainer.append(aui.form.textField({
            labelContent: AJS.I18n.getText('create.issue.form.field.summary.label'),
            isRequired: true,
            name: 'summary',
            value: this._getSummaryFromConfiguration(),
        }));
        defaulFieldsContainer.append(aui.form.textareaField({
            labelContent: AJS.I18n.getText('create.issue.form.field.description.label'),
            name: 'description',
        }));
    };

    JiraCreateIssueForm.prototype._getDefaultFields = function () {
        return {
            server: $('.server-select', this.$form),
            project: $('.project-select', this.$form),
            issueType: $('.issuetype-select', this.$form),
        };
    };

    JiraCreateIssueForm.prototype._loadServerData = function () {
        var me = this;
        this.formIsLoading(this.defaultFields.server);
        JiraCreateIssueFormData.loadServers(me.configuration.ajax)
            .then(function (data) {
                me.formLoadingCompleted(me.defaultFields.server);
                me._loadServerDataComplete(data);
            });
    };

    JiraCreateIssueForm.prototype._loadFields = function (fieldValues) {
        const me = this;
        const context = _.extend({}, this.getContextJSON(), fieldValues);
        this.formIsLoading(this.defaultFields.issueType);
        const fieldsMeta = JiraCreateIssueFormData.getFieldMetadata(context, me.configuration.ajax);

        fieldsMeta.done(
            function (meta) {
                const fields = me._filterFields(meta);

                if (me.configuration.renderSummaryAndDescription) {
                    const descriptionMeta = meta.find(field => field.fieldId === "description");
                    const descriptionIsRequired = !!(descriptionMeta && descriptionMeta.required);
                    me._setDescriptionAsRequiredField(descriptionIsRequired);
                }

                const unrenderableRequiredFields = me._unrenderableRequiredFields(fields);
                if (unrenderableRequiredFields.length === 0) {
                    me._renderFields(fields, context);
                    me._trigger('onFieldsRendered');
                } else {
                    me._handleUnrenderableRequiredFields(unrenderableRequiredFields);
                }

                // provide stats on field types for analytics
                me._fieldTypeAnalytics(fields);

                me.formLoadingCompleted(me.defaultFields.issueType);
            }
        ).fail(_.bind(me._handleAjaxError, this));
    };

    JiraCreateIssueForm.prototype._loadServerDataComplete = function (servers) {
        if (servers.length) {
            if (servers.length === 1) { // hide server fields if we have only one instance
                JiraCreateIssueFormFieldHelper.hideField(this.defaultFields.server);
            }
            this._hasMultiServer = servers.length > 1;
            JiraCreateIssueFormFieldHelper.fillSelectData(this.defaultFields.server, servers,
                this.configuration.serverId || servers[0].id);
            this._selectServer(this.configuration.serverId || servers[0].id);
        } else {
            // Show Jira Server Warning
            this._handlerUnexpectedError(AJS.I18n.getText('create.issue.form.field.server.error.empty'));
        }
    };

    JiraCreateIssueForm.prototype._loadProjectData = function (serverId) {
        var me = this;
        this.formIsLoading(this.defaultFields.project);
        var promise = JiraCreateIssueFormData.loadProjects(serverId, me.configuration.ajax);
        promise.then(
            function (projects) {
                if (projects.errors && projects.errors.length) {
                    var err = projects.errors[0];
                    if (err.authenticationUri) {
                        me._handleAuthorizeError(err);
                    } else {
                        me._handlerUnexpectedError(err.message);
                    }
                } else {
                    if (projects.length) {
                        JiraCreateIssueFormFieldHelper.fillSelectData(me.defaultFields.project,
                            projects, me.configuration.projectId);
                    } else {
                        me._handlerUnexpectedError(AJS.I18n.getText('create.issue.form.field.project.error.empty'));
                    }
                }
                me.formLoadingCompleted(me.defaultFields.project);
            },
            _.bind(me._handleAjaxError, this)
        );
    };

    JiraCreateIssueForm.prototype._loadIssueType = function (serverId, projectId, selectedIssueType) {
        const me = this;
        this.formIsLoading(this.defaultFields.issueType);

        JiraCreateIssueFormData.loadIssueTypes(serverId, projectId, me.configuration.ajax).then(
            function (issueTypes) {
                JiraCreateIssueFormFieldHelper.fillSelectData(
                    me.defaultFields.issueType,
                    issueTypes,
                    selectedIssueType && selectedIssueType.id
                );

                me.formLoadingCompleted(me.defaultFields.issueType);
            }
        ).fail(function (xhr) {
            if (xhr.status === 404) {
                me._handleCommunicationError(AJS.I18n.getText('create.issue.form.error.proxy', 404));
            }

            me.formHasError();
        });
    };

    // TODO if this codebase was modular I would put this function in a separate module
    JiraCreateIssueForm.prototype._fieldTypeAnalytics = function (fields) {
        var fieldData = fields.map(function (field) {
            var restType = JiraIntegrationFields.getFieldType(field);
            return {
                required: field.required,
                restType: restType,
                knownRestType: JiraIntegrationFields.isKnownRestType(restType),
            };
        });

        var attributes = fieldData.reduce(function (memo, value) {
            if (value.knownRestType) {
                memo[value.required ? 'requiredFields' : 'otherFields'].push(value.restType);
            } else {
                memo[value.required ? 'unknownRequiredFieldsCount' : 'unknownOtherFieldsCount']++;
            }
            return memo;
        }, {
            requiredFields: [],
            otherFields: [],
            unknownRequiredFieldsCount: 0,
            unknownOtherFieldsCount: 0,
        });

        // the analytics client library won't send arrays, so we convert them to the format that
        // the backend would anyway if we could send an array.
        //
        // The conversion is this:
        //
        // { foo: ['bar', 'baz'] } => { foo.size: 2, foo[0]: 'bar', foo[1]: 'baz' }
        //
        // This should be a separate helper, but because we're in globals land we'll keep it here so
        // we can test it properly.
        attributes = Object.keys(attributes).reduce(function (memo, key) {
            var value = attributes[key];
            if (!Array.isArray(value)) {
                memo[key] = value;
                return memo;
            }

            var len = value.length;
            memo[key + '.size'] = len;
            for (var i = 0; i < len; i++) {
                memo[key + '[' + i + ']'] = value[i];
            }
            return memo;
        }, {});

        AJS.trigger('analytics', {
            name: 'jira.integration.issue.create.form.displayed',
            data: attributes,
        });
    };

    JiraCreateIssueForm.prototype._renderFields = function (fields, context) {
        this.$containerRequireField.html('');
        var unsupportedFields = _.reject(fields, JiraIntegrationFields.canRender);

        if (!this.configuration.allowUnsupportedFields && unsupportedFields.length) {
            this._handleUnsupportedFields(unsupportedFields);
        } else {
            this.$containerRequireField.html(_.map(fields, function (field) {
                return JiraIntegrationFields.renderField(null, field, context, null);
            }).join(''));
            JiraIntegrationFields.attachFieldBehaviors(this.$containerRequireField, context, null);
            this._trigger('onRequiredFieldsRendered', fields, unsupportedFields);
        }
    };

    JiraCreateIssueForm.prototype._setError = function ($message) {
        removeAuiMessages(this.$form);
        this.$form.prepend($message);
        this.formLoadingCompleted();
        this.formHasError();
    };

    JiraCreateIssueForm.prototype._handleCommunicationError = function (message) {
        var $message = this.configuration.get$communicationError(message);
        this._setError($message);
    };

    JiraCreateIssueForm.prototype._handlerUnexpectedError = function (message) {
        var $message = this.configuration.get$unexpectedError(message);
        this._setError($message);
    };

    JiraCreateIssueForm.prototype._handleUnsupportedFields = function (unsupportFields) {
        unsupportFields = _.map(unsupportFields, function (item) {
            return AJS.escapeHtml(item.name);
        });
        var $message = this.configuration.get$unsupportedFieldMessage(unsupportFields, this._getCreateJiraIssueUrl());
        removeAuiMessages(this.$form);
        this.$form.prepend($message);
        this._trigger('onError');
        this.formLoadingCompleted();
    };

    JiraCreateIssueForm.prototype._handleUnsupportedServer = function (serverUrl) {
        var $message = this.configuration.get$unsupportedServerMessage(serverUrl);
        this._setError($message);
    };

    JiraCreateIssueForm.prototype._handleAjaxError = function (xhr) {
        var me = this;
        if (xhr.status >= 500 && xhr.status < 600) {
            var error = (xhr.responseJSON.errors && xhr.responseJSON.errors[0]);
            var exceptionName = error && error.exceptionName;
            if (exceptionName && exceptionName === 'com.atlassian.integration.jira.JiraCommunicationException') {
                this._handleCommunicationError(error.message);
            } else {
                me._handlerUnexpectedError(AJS.I18n.getText('create.issue.form.error.proxy', xhr.status));
            }
        } else {
            // unexpected response
            me._handlerUnexpectedError(AJS.I18n.getText('communication.unexpected'));
        }
        me.formHasError();
    };

    JiraCreateIssueForm.prototype._handleAuthorizeError = function (data) {
        var me = this;
        this.formHasError();
        var $message = this.configuration.get$unauthorizeMessage(data.applicationName);
        this.$form.append($message);

        $('.applink-authenticate', $message).on('click', function (e) {
            AppLinks.authenticateRemoteCredentials(data.authenticationUri, function () {
                me.resetForm();
                me._selectServer(me.currentServerId, true);
            }, function () {
                me._handlerUnexpectedError(AJS.I18n.getText('create.issue.form.oauth.refused', data.applicationName));
            });
            e.preventDefault();
        });
    };

    JiraCreateIssueForm.prototype._handleUnrenderableRequiredFields = function (fields) {
        var $message = this.configuration.get$unrenderableRequiredFieldsMessage(fields, this._getCreateJiraIssueUrl());
        removeAuiMessages(this.$form);
        this.$form.append($message);
        this.formLoadingCompleted();
    };

    JiraCreateIssueForm.prototype._getCreateJiraIssueUrl = function () {
        var projectId = this.defaultFields.project.find('option:selected').val();
        var issueTypeId = this.defaultFields.issueType.find('option:selected').val();
        var createIssueUrl = JiraCreateIssueFormData.getServerById(this.currentServerId).displayUrl;

        createIssueUrl = createIssueUrl + '/secure/CreateIssueDetails!Init.jspa?pid=' + projectId + '&issuetype=' + issueTypeId;

        var summaryValue = this._getFieldValue('summary');
        if (summaryValue.length) {
            createIssueUrl = createIssueUrl + '&summary=' + encodeURIComponent(summaryValue);
        }

        var descriptionValue = this._getFieldValue('description');
        if (descriptionValue.length) {
            createIssueUrl = createIssueUrl + '&description=' + encodeURIComponent(descriptionValue);
        }

        return createIssueUrl;
    };

    JiraCreateIssueForm.prototype._filterFields = function (fields) {
        var me = this;
        return _.filter(fields, function (field) {
            var typeId = field.schema ? (field.schema.system || field.schema.custom || field.schema.customId) : field;
            var excluded = me.configuration.excludedField && me.configuration.excludedField.includes(typeId) ||
                (me.configuration.ignoreFieldsWithDefaultValue && field.hasDefaultValue) ||
                (me.configuration.requiredFieldsOnly && !field.required);
            return !excluded;
        });
    };

    JiraCreateIssueForm.prototype._unrenderableRequiredFields = function (fields) {
        return fields.filter(function (field) {
            return field.required && !JiraIntegrationFields.canRender(field);
        });
    };

    JiraCreateIssueForm.prototype._getFieldValue = function (fieldName) {
        var $field = $('[name=\'' + fieldName + '\']', this.$form);
        return ($field) ? $.trim($field.val()) : '';
    };

    JiraCreateIssueForm.prototype._setDescriptionAsRequiredField = function (isRequiredField) {
        var descriptionLabel = $('.field-group [name="description"]', this.$form).prev('label');
        descriptionLabel.find('.aui-icon.aui-icon-required').remove(); // try remove first to avoid duplicated

        if (isRequiredField) {
            descriptionLabel.append(aui.icons.icon({icon: 'required'}));
        }
    };

    JiraCreateIssueForm.prototype._getSummaryFromConfiguration = function () {
        // if the summary value is set in the configuration we only want to return it once on the first load
        var summary = this.configuration.initialSummary;
        delete this.configuration.initialSummary;
        return summary;
    };

    JiraCreateIssueForm.prototype._getJSON = function (options) {
        if (!options.verbose) {
            return transform$FieldInputs(options.$form, function ($field, name, i) {
                return options.getFieldJSON($field);
            });
        }

        // Confluence currently pulls all this info out of the DOM. Since I'm writing a function to extract JSON from this,
        // I'm providing them a way to not do the wrong thing anymore.
        var fieldsMeta = JiraCreateIssueFormData.getCachedFieldMetadataEntry(this.getContextJSON()).value;
        var contextsByName = Object.keys(fieldsMeta.fields).reduce(function (o, name) {
            var restField = fieldsMeta.fields[name];
            o[name] = JiraIntegrationFields.getContext(null, restField, null, null);
            return o;
        }, {});
        return transform$FieldInputs(this.$form, function ($field, name, i) {
            var context = contextsByName[name];
            return {
                name: name,
                jiraType: context.jiraType,
                required: context.isRequired,
                label: context.labelText,
                value: options.getFieldJSON($field),
                index: i,
            };
        });
    };

    JiraCreateIssueForm.prototype.resetForm = function (persistFieldValues) {
        removeAuiMessages(this.$form);
        $('.field-group', this.$form).show();
        if (this.configuration.renderSummaryAndDescription) {
            this._setDescriptionAsRequiredField(false);
        }
        if (!this._hasMultiServer) {
            JiraCreateIssueFormFieldHelper.hideField(this.defaultFields.server);
        }
        this.$containerRequireField.html('');
        this.fieldValues = persistFieldValues ? this.fieldValues : {};
    };

    JiraCreateIssueForm.prototype.formHasError = function () {
        $('.field-group', this.$form).hide();
        this.$containerRequireField.html('');
        if (this._hasMultiServer) {
            JiraCreateIssueFormFieldHelper.showField(this.defaultFields.server);
        }
        JiraCreateIssueFormFieldHelper.setFieldDisabled($('.insert-issue-button'), true);
        this._trigger('onError');
        this.formLoadingCompleted();
    };

    JiraCreateIssueForm.prototype.getCurrentServer = function () {
        return JiraCreateIssueFormData.getServerById(this.currentServerId);
    };

    JiraCreateIssueForm.prototype.formIsLoading = function ($byField) {
        if ($byField) {
            JiraCreateIssueFormFieldHelper.setIsLoading($byField, true);
        }
        var allFields = $(':input', $(this.$form));
        JiraCreateIssueFormFieldHelper.setFieldDisabled(allFields, true);
    };

    JiraCreateIssueForm.prototype.formLoadingCompleted = function ($byField) {
        if ($byField) {
            JiraCreateIssueFormFieldHelper.setIsLoading($byField, false);
        } else {
            $('.aui-icon.aui-icon-wait', this.$form).remove();
        }
        var allFields = $(':input', $(this.$form));
        JiraCreateIssueFormFieldHelper.setFieldDisabled(allFields, false);
    };

    JiraCreateIssueForm.prototype.createSelect2WithIcon = function ($element) {
        if ($element.is('.server-select')) {
            $element.auiSelect2({'minimumResultsForSearch': -1});
        } else {
            var select2Config = $element.is('.project-select') ? {
                'formatSelection': this.projectSelectFormat,
                'formatResult': this.projectSelectFormat,
                'extraAttributes': this.defaultFields,
            } : {
                'formatSelection': this.issueTypeSelectFormat,
                'formatResult': this.issueTypeSelectFormat,
                'minimumResultsForSearch': -1,
                'extraAttributes': this.defaultFields,
            };
            $element.auiSelect2(select2Config);
        }
        $element.auiSelect2('val', '');
    };

    JiraCreateIssueForm.prototype.projectSelectFormat = function (option) {
        var serverId = this.extraAttributes.server.select2('data').id;
        var imageUrl = JiraCreateIssueFormData.getProjectIconUrl(serverId, option.id);
        return templates.fields.select2WithIconOption({
            'optionValue': option.text,
            'iconUrl': imageUrl,
            'isProject': true,
            "lazyLoadImage": true
        });
    };

    JiraCreateIssueForm.prototype.issueTypeSelectFormat = function (option) {
        var serverId = this.extraAttributes.server.select2('data').id;
        var projectId = this.extraAttributes.project.select2('data').id;
        var imageUrl = JiraCreateIssueFormData.getIssueTypeIconUrl(serverId, projectId, option.id);
        return templates.fields.select2WithIconOption({
            'optionValue': option.text,
            'iconUrl': imageUrl,
            'isProject': false,
        });
    };

    JiraCreateIssueForm.prototype.getContextJSON = function () {
        var projectId = this.defaultFields.project.val() || $(this.defaultFields.project).select2('data').value;
        return {
            serverId: this.currentServerId,
            projectId: projectId,
            projectKey: JiraCreateIssueFormData.getProjectById(
                this.currentServerId,
                projectId
            ).key,
            issueTypeId: this.defaultFields.issueType.val(),
            summary: this._getSummaryFromConfiguration(),
        };
    };

    /**
     * Get an object keyed by input name for each field in the form.
     * @param {boolean} verbose - true if you want metadata about the field (like type, label, whether it's required); false to get the JSON value only.
     * @returns {{}}
     */
    JiraCreateIssueForm.prototype.getJSON = function (verbose) {
        return JiraCreateIssueForm.prototype._getJSON({
            getFieldJSON: JiraIntegrationFields.getJSON,
            verbose: verbose,
            $form: this.$form,
        });
    };

    /**
     * Renders a non-field related error message at the top of the create jira issue form
     */
    JiraCreateIssueForm.prototype.renderUnexpectedError = function (message) {
        var $message = this.configuration.get$unexpectedError(message);
        removeAuiMessages(this.$form);
        this.$form.prepend($message);
        this.formLoadingCompleted();
    };

    /**
     * Renders errors associated with the form fields
     */
    JiraCreateIssueForm.prototype.renderErrors = function (errors) {
        transform$FieldInputs(this.$form, function ($fieldInput, name) {
            var $field = $fieldInput.closest('.jira-field');
            var startsWithFieldName = function (element) {
                return element.indexOf(name) === 0;
            };
            var errorArray;
            if (errors.hasOwnProperty(name) || Object.keys(errors).some(startsWithFieldName)) {
                if (!errors.hasOwnProperty(name)) {
                    name = Object.keys(errors).filter(startsWithFieldName)[0];
                }
                errorArray = Array.isArray(errors[name]) ? errors[name] : [errors[name]];
            }
            JiraIntegrationFields.setFieldError($field, errorArray);
        });
    };

    JiraCreateIssueForm.prototype.submit = function () {
        this.$form.submit();
    };

    function transform$FieldInputs ($form, mapper) {
        var transformed = {};
        $form
            .find('.create-issue-required-fields')
            .find('input, select, textarea, fieldset')
            .not('.select2-input, .select2-focusser')
            .each(function (i) {
                var $field = $(this);
                var name = $field.attr('data-name');
                if (name) {
                    transformed[name] = mapper($field, name, i);
                }
            });
        return transformed;
    }

    function removeAuiMessages ($el) {
        $('.aui-message', $el).remove();
    }

    return JiraCreateIssueForm;
});
