'use strict';

define('bitbucket-plugin-tag/internal/feature/tags/commit-tags/commit-tags', ['@atlassian/aui', 'jquery', 'lodash', 'bitbucket/util/navbuilder', 'bitbucket/util/server', 'bitbucket/util/state', 'bitbucket/internal/model/page-state', 'bitbucket/internal/util/analytics', 'bitbucket/internal/util/feature-detect'], function (AJS, $, _, nav, server, pageState, modelPageState, analytics, feature) {
    'use strict';

    var tags = [];
    var hasRepoWritePermission;
    var lastTagNameCheck;

    var $tagInlineDialog;
    var $tagNameGroup;

    var TAG_TYPE_ANNOTATED = 'ANNOTATED';
    var TAG_TYPE_LIGHTWEIGHT = 'LIGHTWEIGHT';
    var CreateDialogStates = {
        SUCCESS: 'SUCCESS',
        ERROR: 'ERROR',
        TYPING: 'TYPING',
        RESET: 'RESET'
    };

    function buildGetTagsUrl(project, repository) {
        return nav.rest('tags').addPathComponents('projects', project.getKey(), 'repos', repository.getSlug(), 'tags').build();
    }

    function buildViewTagUrl(tag) {
        return nav.repository(pageState.getRepository()).commits().until(tag).build();
    }

    function stopScrollJumpOnTagLoad($tagPlaceholder) {
        window.scrollBy(0, $tagPlaceholder.height());
    }

    function enrichTagsWithUrl(newTags) {
        return newTags.map(function (tag) {
            tag.url = buildViewTagUrl(tag);

            return tag;
        });
    }

    function renderTags() {
        $('.delete-tag-dialog').remove();
        var $newContent = $(bitbucketPluginTag.internal.feature.tags.commitTags.commitTags.tagInfo({
            tags: tags,
            canAddAndDeleteTags: hasRepoWritePermission
        }));
        $newContent.replaceAll('.plugin-item.tags .tags-info');
    }

    function getTagByDialogId(dialogId) {
        return $('.tag-lozenge[aria-controls=' + dialogId + ']');
    }

    function removeTag() {
        var $deleteDialog = $(this).closest('.delete-tag-dialog');
        var tagName = $deleteDialog.attr('data-tag-name');
        $deleteDialog.removeAttr('open');

        // optimistically hide the deleted tag
        var $tag = getTagByDialogId($deleteDialog.attr('id'));
        var animationPromise = $.Deferred();
        $tag.addClass('hidden-tag').one(feature.transitionEndEventName(), function () {
            animationPromise.resolve();
        });
        $('#tags-see-more').click();

        var navPath = nav.rest('git').repository(pageState.getRepository()).addPathComponents('tags');
        var requestPromise = server.rest({
            url: navPath.addPathComponents.apply(navPath, tagName.split('/')).build(),
            type: 'DELETE',
            statusCode: {
                404: function _() {
                    // tag has already been deleted but we don't want to show the 404 error message
                    // return a resolved promise so the done runs as expected
                    return $.Deferred().resolve();
                },
                400: function _(xhr, textStatus, errorThrown, _ref) {
                    var errors = _ref.errors;

                    addErrorNotifications(errors);

                    return false;
                }
            }
        }).done(function () {
            tags = tags.filter(function (tag) {
                return tag.displayId !== tagName;
            });
            analytics.add('commit.tags.deleted');
        });

        $.when(requestPromise, animationPromise).always(renderTags);
    }

    function addErrorNotifications(errors) {
        if (errors && errors.length > 0) {
            var errorBody = errors.reduce(function (result, error) {
                result += bitbucket.internal.widget.exception.exception.errorContent(error);

                return result;
            }, '');

            AJS.flag({
                type: 'error',
                body: errorBody
            });
        }
    }

    function initialiseDeleteTag() {
        $(document).on('click', '.delete-tag-dialog .aui-button', removeTag);

        $(document).on('aui-show', '.delete-tag-dialog', function () {
            getTagByDialogId(this.id).addClass('active');
        }).on('aui-hide', '.delete-tag-dialog', function () {
            getTagByDialogId(this.id).removeClass('active');
        });

        $('.plugin-item.tags').on('focusin', '.aui-label-split-close', function () {
            $(this).closest('.tag-lozenge').addClass('active');
        }).on('focusout', '.aui-label-split-close', function () {
            var $tag = $(this).closest('.tag-lozenge');

            if ($tag.attr('aria-expanded') !== 'true') {
                $tag.removeClass('active');
            }
        }).on('keydown', '.aui-label-split-close', function (e) {
            if (e.which === AJS.keyCode.ENTER) {
                $(this).closest('.tag-lozenge').click();
            }
        });
    }

    function initialiseTags(initialTags) {
        if (initialTags.length) {
            analytics.add('commit.tags.rendered', {
                count: initialTags.length
            });
        }
        tags = enrichTagsWithUrl(initialTags);
        var $tagPlaceholder = $(bitbucketPluginTag.internal.feature.tags.commitTags.commitTags.tags({
            tags: tags,
            canAddAndDeleteTags: hasRepoWritePermission
        })).replaceAll('.plugin-item.tags');

        $('.plugin-item.tags').on('click', '#tags-see-more', function () {
            $(this).closest('.plugin-item.tags').removeClass('hide-more');
        });

        stopScrollJumpOnTagLoad($tagPlaceholder);

        $(document).tooltip({
            live: '.tag-lozenge a, a.tag-lozenge',
            title: function title() {
                return $(this).text();
            }
        });

        $('.plugin-item.tags').on('click', '#tags-see-more', function (e) {
            e.preventDefault();
        });

        initialiseCreateTag();
        initialiseDeleteTag();
    }

    var checkTagAvailable = function checkTagAvailable(tagName) {
        tagName = tagName.trim();
        var navPath = nav.rest().repository(pageState.getRepository()).addPathComponents('tags');
        lastTagNameCheck = tagName;
        server.rest({
            url: navPath.addPathComponents.apply(navPath, tagName.split('/')).build(),
            statusCode: {
                400: function _() {
                    // bad request - user tried to use characters that are invalid for a url
                    // (such as a backslash) give them a helpful error message instead
                    setCreateDialogState(CreateDialogStates.ERROR, AJS.I18n.getText('bitbucket.plugin.tag.create.tag.error.invalid'));

                    return false;
                },
                404: function _() {
                    // if tag is not duplicated and request succeed try to submit the new tag
                    submitNewTag();

                    return false;
                }
            }
        }).done(function () {
            setCreateDialogState(CreateDialogStates.ERROR, AJS.I18n.getText('bitbucket.plugin.tag.create.tag.error.exists'));
        });
    };

    function setCreateDialogState(state, errorMessage) {
        $tagNameGroup.toggleClass('error', state === CreateDialogStates.ERROR);
        clearErrors();

        if (state === CreateDialogStates.ERROR) {
            var errorHtml = $('<div class="error">\n                <ul>\n                    <li><span class="aui-icon aui-icon-small aui-iconfont-error aui-icon-notification"></span>' + errorMessage + '</li>\n                </ul>\n            </div>');

            $tagNameGroup.find('.field-group').append(errorHtml);
        }

        if (state === CreateDialogStates.RESET) {
            $tagInlineDialog.find('input.text').val('');
        }
    }

    function initialiseCreateTag() {
        $tagInlineDialog = $('#create-tag-inline-dialog');
        $tagNameGroup = $tagInlineDialog.find('.tag-name-group');
        var $newTagName = $tagInlineDialog.find('#new-tag-name');

        $newTagName.on('input', function () {
            // if user is changing the input clear the errors
            lastTagNameCheck = null;
            clearErrors();
            setCreateDialogState();
        });

        $tagInlineDialog.find('form').on('submit', function (e) {
            e.preventDefault();
            var tagName = $newTagName.val();
            checkTagAvailable(tagName);
        });

        $tagInlineDialog.find('.aui-iconfont-error').tooltip({
            delayIn: 0,
            gravity: 'ne'
        });

        $(document).on('aui-show', '#create-tag-inline-dialog, .delete-tag-dialog', function (e) {
            setCreateDialogState(CreateDialogStates.RESET);
            _.defer(function () {
                $(e.target).find('input, button').first().focus();
            });
        });
    }

    function clearErrors() {
        $tagNameGroup.find('.error').remove();
        $tagInlineDialog.find('.aui-message-error').remove();
    }

    function setErrorsMessage(errors) {
        var errorBody = errors.reduce(function (result, error) {
            result += bitbucketPluginTag.internal.feature.tags.commitTags.commitTags.errorContent({
                error: error,
                extraClasses: errors.length === 1 && 'single-error'
            });

            return result;
        }, '');

        clearErrors();

        $tagNameGroup.prepend(AJS.messages.error({
            title: _.get(errors, '0.message'),
            body: errorBody,
            closeable: false
        }));
    }

    function submitNewTag() {
        function handleError(xhr, textStatus, errorThrown, _ref2) {
            var errors = _ref2.errors;

            setErrorsMessage(errors);

            return false;
        }

        var message = $tagInlineDialog.find('#tag-description').val();
        server.rest({
            url: nav.rest('git').repository(pageState.getRepository()).addPathComponents('tags').build(),
            type: 'POST',
            data: {
                message: message,
                name: $tagNameGroup.find('input').val().trim(),
                startPoint: modelPageState.getCommit().getId(),
                type: message.length > 0 ? TAG_TYPE_ANNOTATED : TAG_TYPE_LIGHTWEIGHT
            },
            statusCode: {
                400: handleError,
                409: handleError
            }
        }).done(function (newTag) {
            setCreateDialogState(CreateDialogStates.RESET);
            $tagInlineDialog.removeAttr('open');

            newTag.url = buildViewTagUrl(newTag);
            tags.push(newTag);
            renderTags();
            lastTagNameCheck = null;
            analytics.add('commit.tags.added', {
                annotated: message.length > 0
            });
        });
    }

    return {
        fetchTagsForCommit: function fetchTagsForCommit(options) {
            hasRepoWritePermission = options.hasRepoWrite;
            server.rest({
                data: [modelPageState.getCommit().getId()],
                type: 'POST', //POST to support an arbitrary number of commit IDs (each ID being 40 characters)
                url: buildGetTagsUrl(modelPageState.getProject(), modelPageState.getRepository()),
                statusCode: {
                    500: false,
                    0: function _(xhr, statusText) {
                        if (statusText === 'timeout') {
                            analytics.add('commit.tags.fetchTimedout');
                        }

                        return false;
                    }
                }
            }).done(function (response) {
                initialiseTags(response.values);
            });
        }
    };
});