/* Custom scripts for Embedded Crowd administration */
(function () {
    var $ = require('jquery');
    const wrmContextPath = require('wrm/context-path');

    function getField($form, name) {
        return $("#" + $form.attr("id") + "-" + name, $form);
    }

    function setFieldEnabled(el, enabled) {
        if (enabled) {
            $(el).prop("disabled", false);
        } else {
            $(el).prop("disabled", true);
        }
    }

    function setFieldValue(el, value) {
        if ($(el).attr("type") == "checkbox") {
            $(el).prop("checked", value == "true");
        } else {
            $(el).val(value);
        }
    }

    // get the directory config for the selected directory type from ldapTypes, found in a script tag in the body
    function getConfiguration(directoryType) {
        return typeof(ldapTypes) != "undefined" && ldapTypes[directoryType];
    }

    var fieldsWithDefaultsSet = []; // has to persist across calls to updateDefaults()
    function applyFieldDefaults($form, directoryType) {
        var config = getConfiguration(directoryType);
        if (!config) return;

        // clear all the previously set fields
        for (var i=0; i < fieldsWithDefaultsSet.length; i++) {
            setFieldValue(fieldsWithDefaultsSet[i], "");
        }
        fieldsWithDefaultsSet = [];

        // set the fields to their default values found in the configuration for this directory type
        for (var key in config.defaults) {
            if (config.defaults.hasOwnProperty(key)) {
                var id = key.replace(/-(.)/g, function () { return arguments[1].toUpperCase() });
                var $field = getField($form, id);
                if ($field.length && !$field.hasClass("checkbox-overridden-globally")) {
                    setFieldValue($field, config.defaults[key]);
                    fieldsWithDefaultsSet.push($field);
                }
            }
        }
        // Make sure the enablement of pagedResults is correct.
        var $pagedResults = getField($form, "ldapPagedresults");
        var $pageSize = getField($form, "ldapPagedresultsSize");
        setFieldEnabled($pageSize, $pagedResults.is(":checked"));
    }

    function applyHiddenFields($form, directoryType) {
        var config = getConfiguration(directoryType),
            id;
        if (!config) return;

        // show all previously hidden fields
        $(".hideable", $form).removeClass("hidden");

        // hide fields declared to be hidden in the configuration for this directory type
        for (var i=0; i< config.hidden.length; i++) {
            id = config.hidden[i].replace(/-(.)/g, function () { return arguments[1].toUpperCase() });
            getField($form, id).closest(".hideable").addClass('hidden');
        }
    }

    function toggleFieldBasedOnRadioButtonValue($form, inputName, fieldToToggleId, value) {
        var $option = $("input[name=" + inputName + "]", $form);
        function toggle() {
            $(fieldToToggleId, $form).toggleClass("hidden", !($option.filter(":checked").val() === value));
        }
        $option.change(toggle);
        toggle();
    }

    $(document).ready(function () {
        var $form = $("form", "#embcwd");
        var $type = getField($form, "type");
        var $placeholderForm = $("#embcwd-placeholder-form");

        // enable or disable "page size" field based on whether "paged results" is checked
         var $pagedResults = getField($form, "ldapPagedresults");
         var $pageSize = getField($form, "ldapPagedresultsSize");
         $pagedResults.change(function () {
             setFieldEnabled($pageSize, $pagedResults.is(":checked"));
         });
         $form.bind("changed-type.embcwd", function () {
             setFieldEnabled($pageSize, $pagedResults.is(":checked"));
         });

        function isNewForm() {
            var directoryId = getField($form, "directoryId").val();
            var newForm = getField($form, "newForm").val();
            return (directoryId == "" || directoryId == "0") && (newForm == "" || newForm == "true");
        }

        function setNewForm(value) {
            var newForm = getField($form, "newForm");
            setFieldValue(newForm, value);
        }

        if (isNewForm()) {
            // if this is a new directory, we want to set default values based on Crowd's settings when changing types
            $form.bind("changed-type.embcwd", function () {
                applyFieldDefaults($(this), $type.val());
            });
            setNewForm("false");
        }
        // however, we always want to hide unnecessary fields, regardless of whether we're in edit or create mode
        $form.bind("changed-type.embcwd", function () {
            applyHiddenFields($(this), $type.val());
            $form.attr("test-last-set-directory-type", $type.val());
        });

        // trigger stuff that needs updating whenever the directory type is changed
        $form.trigger("changed-type.embcwd");
        $type.change(function () {
            $form.trigger("changed-type.embcwd");
        });

        var $useSSLField = getField($form, "useSSL"),
            $secureSSLField = getField($form, "ldapSecure"),
            $portField = getField($form, "port");

        // Set initial state of "secure SSL" checkbox
        if (!$useSSLField.is(":checked")) {
            $secureSSLField.prop('disabled', true);
        }
        // Handle checking/unchecking of SSL option
        $useSSLField.change(function () {
            if ($(this).is(":checked")) {
                $portField.val("636");
                $secureSSLField.prop({
                    disabled: false,
                    checked: true
                });
            } else {
                $portField.val("389");
                $secureSSLField.prop({
                    disabled: true,
                    checked: false
                });
            }
        });

        var $synchroniseGroupMemberships = getField($form, "synchroniseGroupMemberships"),
            isDelegatedLdapDirectory = $synchroniseGroupMemberships.length > 0;

        if (isDelegatedLdapDirectory) {
            function toggleSynchroniseGroupMembershipOptions() {
                $("#toggle-group-schema-settings").toggleClass("hidden", !$synchroniseGroupMemberships.is(":checked"));
                $("#toggle-membership-schema-settings").toggleClass("hidden", !$synchroniseGroupMemberships.is(":checked"));
            }
            $synchroniseGroupMemberships.change(toggleSynchroniseGroupMembershipOptions);
            toggleSynchroniseGroupMembershipOptions();

            // User Schema Settings, updateUserOnAuth and ldapAutoAddGroups are displayed only if
            // createUserOnAuth is checked in delegated ldap screen.
            var $createUserOnAuth = getField($form, "createUserOnAuth");
            function toggleCreateUserOnAuthOptions() {
                var $ldapAutoAddGroups = getField($form, "ldapAutoAddGroups"),
                    $updateUserOnAuth =  getField($form, "updateUserOnAuth"),
                    createUserOnAuthChecked = $createUserOnAuth.is(":checked");

                $createUserOnAuth.parent().find(".field-group").toggleClass("disabled-group", !createUserOnAuthChecked);

                if (createUserOnAuthChecked) {
                    $ldapAutoAddGroups.prop("disabled", false);
                    $updateUserOnAuth.prop("disabled", false);
                    $synchroniseGroupMemberships.prop("disabled", false);
                } else {
                    $synchroniseGroupMemberships.prop("checked", false);
                    toggleSynchroniseGroupMembershipOptions();
                    $synchroniseGroupMemberships.prop("disabled", true);
                    $ldapAutoAddGroups.prop("disabled", true);
                    $updateUserOnAuth.prop("checked", false);
                    $updateUserOnAuth.prop("disabled", true);
                }
            }
            $createUserOnAuth.change(toggleCreateUserOnAuthOptions);
            toggleCreateUserOnAuthOptions();
        } else {
            // ldapAutoAddGroups is displayed only if ldap directory has read only local groups permission.
            toggleFieldBasedOnRadioButtonValue($form, "ldapPermissionOption", "#ldap-auto-add-groups-field-group", "READ_ONLY_LOCAL_GROUPS");
        }

        // ldapSpringConfig is displayed only if the commons pool 2 pooling type is selected.
        toggleFieldBasedOnRadioButtonValue($form, "ldapPoolType", "#ldap-spring-config-field-group", "COMMONS_POOL2");

        // section toggling
        $(".toggle-section", $form).each(function () {
            var $section = $(this);
            var $icon = $section.find(".toggle-head .aui-icon");
            var $body = $section.find(".toggle-body");

            if ($body.find(".error").length > 0) {
                $section.removeClass("collapsed"); // show sections with validation errors
                $icon.addClass("aui-iconfont-expanded").removeClass("aui-iconfont-collapsed");
            }
            $section.find(".toggle-head").click(function () {
                if ($section.hasClass("updating")) return;
                $section.addClass("updating");
                if ($section.hasClass("collapsed")) {
                    $section.removeClass("collapsed updating");
                    $icon.addClass("aui-iconfont-expanded").removeClass("aui-iconfont-collapsed");
                } else {
                    $section.addClass("collapsed").removeClass("updating");
                    $icon.addClass("aui-iconfont-collapsed").removeClass("aui-iconfont-expanded");
                }
            });
        });

        $("[data-method]").each(function() {
            wirePlaceholderFormButton($(this));
        })

        var startTimer = function (id, xsrfTokenValue) {
            var intervalId = window.setInterval(function () {
                // Jira, Confluence and AUI use different methods to supply the context path AAhhhhhh!
                var path;
                if (AJS.General) {
                    path = AJS.General.getContextPath();
                } else if (AJS.contextPath) {
                    path = AJS.contextPath();
                } else if (contextPath) {
                    path = contextPath;
                }

                $.ajax({
                    "url": path + "/rest/crowd/1/directory/" + id,
                    "contentType": "application/json",
                    "dataType": "json",
                })
                    .done(function (data) {
                        const {currentStartTime, currentDuration, lastStartTime, lastDuration, syncStatus} = data.synchronisation;

                        $(".sync-state.directory-"+id).text(syncStatus);

                        if (currentStartTime == 0) {
                            $(".sync-info.directory-" + id).replaceWith(
                                Crowd.Templates.Private.Directories.Synchronisation.complete({
                                    id,
                                    lastStartTime,
                                    lastDuration
                                }));
                            $(".sync-status.directory-" + id).replaceWith(Crowd.Templates.Private.Directories.Synchronisation.link({
                                id,
                                url: `${wrmContextPath()}/plugins/servlet/embedded-crowd/directories/sync?directoryId=${id}`
                            }));
                            wirePlaceholderFormButton($("a.directory-operation.sync[data-id=" + id + "]"));
                            window.clearTimeout(intervalId);
                        } else {
                            $(".sync-status.directory-" + id).replaceWith(
                                Crowd.Templates.Private.Directories.Synchronisation.progress({
                                    id,
                                    seconds: currentDuration,
                                }));
                        }
                    })
                    .fail(function () {
                        $(".sync-state.directory-" + id).replaceWith(
                            Crowd.Templates.Private.Directories.Synchronisation.failed({id})
                        );
                        $(".sync-status.directory-" + id).replaceWith(
                            Crowd.Templates.Private.Directories.Synchronisation.link({
                                id,
                                url: `${wrmContextPath()}/plugins/servlet/embedded-crowd/directories/sync?directoryId=${id}`
                            }));
                        wirePlaceholderFormButton($("a.directory-operation.sync[data-id=" + id + "]"));
                        window.clearTimeout(intervalId);
                    })
                ;
            }, 1000);
        };

        function wirePlaceholderFormButton ($button) {
            $button.click(function () {
                var method = $button.attr("data-method") ? $button.attr("data-method") : '';
                if (method.toUpperCase() !== 'GET') {
                    $placeholderForm.attr("action", $button.attr("href"));
                    $placeholderForm.attr("method", method);
                    $placeholderForm.submit();
                    return false;
                }
            });
        }

        $(".sync-status.in-progress").each(function () {
            var op = $(this);
            var id = op.attr("data-id");
            op.replaceWith(Crowd.Templates.Private.Directories.Synchronisation.started({id}));
            startTimer(id);
        });

        //Create password placeholders for inputs
        function generatePasswordPlaceHolderInput(passwordInput){
            var passwordPlaceHolder = passwordInput.clone();
            passwordPlaceHolder.removeClass("password_value_placeholder");
            passwordPlaceHolder.attr('id', passwordPlaceHolder.attr('id') + "-placeHolder");
            passwordPlaceHolder.attr('name', null);
            passwordPlaceHolder.prop('value', "password_value_placeholder");
            return passwordPlaceHolder;
        }

        function createPasswordPlaceHolder(passwordInput) {
            var passwordPlaceHolder = generatePasswordPlaceHolderInput(passwordInput);
            passwordInput.after(passwordPlaceHolder);

            passwordPlaceHolder.on('focus', function(){
                passwordPlaceHolder.hide();
                passwordInput.show();
                passwordInput.focus();
            });

            passwordInput.on('blur', function(){
                if(!passwordInput.val()) {
                    passwordInput.hide();
                    passwordPlaceHolder.show();
                }
            });

            passwordInput.hide();
        }


        function createPasswordPlaceHolders() {
            AJS.$('input.password_value_placeholder').each(function (){
                createPasswordPlaceHolder(AJS.$(this));
            });
        }
        createPasswordPlaceHolders();
    });

    // event handler for help links
    document.addEventListener('DOMContentLoaded', function () {
        Array.from(document.getElementsByClassName("help-link"))
            .forEach(function (helpLink) {
                helpLink.addEventListener('click', function(event) {
                    event.preventDefault(); // Prevents navigation
                    const url = this.getAttribute('href');
                    const child = window.open(url, 'localHelp', 'width=600, height=500, resizable, scrollbars=yes');
                    if (child) {
                        child.focus();
                    }
                });
            });
    });
})();
