(function (dcBrowserMetrics, browserMetricsAaBeacon) {
    'use strict';

    const isNumber = (value) => {
        return typeof value === "number";
    };

    function isApdexReport(report) {
        return isNumber(report["apdex"]);
    }
    const apdexReportMarshaller = report => {
        if (isApdexReport(report)) {
            return { apdex: "" + report.apdex };
        }
    };

    const isBoolean = (value) => {
        return typeof value === "boolean";
    };

    const isString = (value) => {
        return typeof value === "string";
    };

    const defaultsReportMarshaller = report => {
        var marshalled = {};
        Object.keys(report).forEach(key => {
            var value = report[key];
            if (isNumber(value) ||
                isBoolean(value) ||
                isString(value) ||
                value === null) {
                marshalled[key] = value;
            }
        });
        return marshalled;
    };

    const marshalTime = (value) => {
        return "" + Math.floor(value);
    };

    function isFirstPaintReport(report) {
        return isNumber(report["firstPaint"]);
    }
    const firstPaintReportMarshaller = report => {
        if (isFirstPaintReport(report)) {
            return { firstPaint: marshalTime(report.firstPaint) };
        }
    };

    function isIsInitialReport(report) {
        return typeof report["isInitial"] === "boolean";
    }
    const isInitialReportMarshaller = report => {
        if (isIsInitialReport(report)) {
            return { isInitial: "" + report.isInitial };
        }
    };

    function isJourneyIdReport(report) {
        return isString(report["journeyId"]);
    }
    const journeyIdReportMarshaller = report => {
        if (isJourneyIdReport(report)) {
            return { journeyId: report.journeyId };
        }
    };

    function isKeyReport(report) {
        return isString(report["key"]);
    }
    const keyReportMarshaller = report => {
        if (isKeyReport(report)) {
            return { key: report.key };
        }
    };

    function isNavigationTypeReport(report) {
        return isNumber(report["navigationType"]);
    }
    const navigationTypeReportMarshaller = report => {
        if (isNavigationTypeReport(report)) {
            return { navigationType: "" + report.navigationType };
        }
    };

    function isReadyForUserReport(report) {
        return isNumber(report["readyForUser"]);
    }
    const readyForUserReportMarshaller = report => {
        if (isReadyForUserReport(report)) {
            return { readyForUser: marshalTime(report.readyForUser) };
        }
    };

    function isRedirectCountReport(report) {
        return isNumber(report["redirectCount"]);
    }
    const redirectCountReportMarshaller = report => {
        if (isRedirectCountReport(report)) {
            return { redirectCount: "" + report.redirectCount };
        }
    };

    function isResourceLoadedEndReport(report) {
        return isNumber(report["resourceLoadedEnd"]);
    }
    const resourceLoadedEndReportMarshaller = report => {
        if (isResourceLoadedEndReport(report)) {
            return { resourceLoadedEnd: marshalTime(report.resourceLoadedEnd) };
        }
    };

    const forEachKeyValue = (object, callback) => {
        Object.keys(object).forEach(key => {
            callback(key, object[key]);
        });
    };

    function len(obj) {
        return Object.keys(obj).length;
    }
    class TrieMultiValueMap {
        constructor() {
            this.children = {};
            this.values = [];
        }
        add(key, value) {
            if (key.length == 0) {
                this.values.push(value);
            }
            else {
                if (!this.children.hasOwnProperty(key[0])) {
                    this.children[key[0]] = new TrieMultiValueMap();
                }
                this.children[key[0]].add(key.substr(1), value);
            }
        }
        toTrieObject() {
            let childrenTrieObject = {};
            forEachKeyValue(this.children, (key, child) => {
                childrenTrieObject[key] = child.toTrieObject();
            });
            if (this.values.length === 0) {
                return childrenTrieObject;
            }
            if (len(this.children) === 0) {
                return this.values;
            }
            return [childrenTrieObject, this.values];
        }
        condensed() {
            let dense = new TrieMultiValueMap();
            dense.values = this.values.concat();
            forEachKeyValue(this.children, (key, child) => {
                let denseChild = child.condensed();
                if (denseChild.values.length === 0 &&
                    len(denseChild.children) === 1) {
                    forEachKeyValue(denseChild.children, (key2, child2) => {
                        dense.children[key + key2] = child2;
                    });
                }
                else {
                    dense.children[key] = denseChild;
                }
            });
            return dense;
        }
    }

    function isResourceTimingReport(report) {
        return Array.isArray(report["resourceTiming"]);
    }
    const INITIATOR_TYPES = {
        other: 0,
        img: 1,
        link: 2,
        script: 3,
        css: 4,
        xmlhttprequest: 5,
    };
    const INVALID_INITIATOR_TYPE = -1;
    function encodedTiming(time) {
        time = Math.floor(time || 0);
        return time > 0 ? toBase36(time) : "";
    }
    function toBase36(n) {
        return n.toString(36);
    }
    function serialiseResourceTiming(resource) {
        return [
            INITIATOR_TYPES[resource.initiatorType] || INVALID_INITIATOR_TYPE,
            encodedTiming(resource.startTime),
            encodedTiming(resource.responseEnd),
            encodedTiming(resource.responseStart),
            encodedTiming(resource.requestStart),
            encodedTiming(resource.connectEnd),
            encodedTiming(resource.secureConnectionStart),
            encodedTiming(resource.connectStart),
            encodedTiming(resource.domainLookupEnd),
            encodedTiming(resource.domainLookupStart),
            encodedTiming(resource.redirectEnd),
            encodedTiming(resource.redirectStart),
        ]
            .join(",")
            .replace(/,+$/, "");
    }
    const resourceTimingReportMarshaller = report => {
        if (isResourceTimingReport(report)) {
            let serialisedResourceTiming = new TrieMultiValueMap();
            report.resourceTiming.forEach(resource => {
                let url = browserMetricsAaBeacon.cleanUrl(resource.name);
                serialisedResourceTiming.add(url || "☠", serialiseResourceTiming(resource));
            });
            return {
                resourceTiming: JSON.stringify(serialisedResourceTiming.condensed().toTrieObject()),
            };
        }
    };

    function isThresholdReport(report) {
        return isNumber(report["threshold"]);
    }
    const thresholdReportMarshaller = report => {
        if (isThresholdReport(report)) {
            return { threshold: marshalTime(report.threshold) };
        }
    };

    let attrs = "unloadEventStart,unloadEventEnd,redirectStart,redirectEnd,fetchStart,domainLookupStart,domainLookupEnd,connectStart,connectEnd,secureConnectionStart,requestStart,responseStart,responseEnd,domLoading,domInteractive,domContentLoadedEventStart,domContentLoadedEventEnd,domComplete,loadEventStart,loadEventEnd".split(",");
    const timingReportMarshaller = report => {
        let properties = {};
        attrs.forEach(attr => {
            if (isNumber(report[attr])) {
                properties[attr] = marshalTime(report[attr]);
            }
        });
        return properties;
    };

    function isUserAgentReport(report) {
        return isString(report["userAgent"]);
    }
    const userAgentReportMarshaller = report => {
        if (isUserAgentReport(report)) {
            return { userAgent: report.userAgent };
        }
    };

    function isUserTimingReport(report) {
        return typeof report["marks"] === "object";
    }
    const userTimingReportMarshaller = report => {
        if (isUserTimingReport(report)) {
            let properties = {};
            let userTimingsRaw = {
                marks: {},
                measures: {},
            };
            report.marks.forEach(mark => {
                userTimingsRaw.marks[mark.name] =
                    userTimingsRaw.marks[mark.name] || [];
                userTimingsRaw.marks[mark.name].push(Math.floor(mark.time));
            });
            forEachKeyValue(userTimingsRaw.marks, (name, times) => {
                properties["mark." + name] = JSON.stringify(times);
            });
            report.measures.forEach(measure => {
                userTimingsRaw.measures[measure.name] =
                    userTimingsRaw.measures[measure.name] || [];
                userTimingsRaw.measures[measure.name].push([
                    Math.floor(measure.startTime),
                    Math.floor(measure.duration),
                ]);
            });
            forEachKeyValue(userTimingsRaw.measures, (name, startTimeAndDurations) => {
                properties["measure." + name] = JSON.stringify(startTimeAndDurations);
            });
            properties.userTimingRaw = JSON.stringify({
                marks: userTimingsRaw.marks,
                measures: userTimingsRaw.measures,
            });
            return properties;
        }
    };

    const elementTimingsReportMarshaller = report => {
        return { elementTimings: JSON.stringify(report.elementTimings) };
    };

    browserMetricsAaBeacon.addReportMarshaller(defaultsReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(elementTimingsReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(apdexReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(firstPaintReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(isInitialReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(journeyIdReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(keyReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(navigationTypeReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(readyForUserReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(redirectCountReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(resourceLoadedEndReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(resourceTimingReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(thresholdReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(timingReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(userAgentReportMarshaller);
    browserMetricsAaBeacon.addReportMarshaller(userTimingReportMarshaller);
    const aaBeaconSubscriber = (beaconData) => {
        browserMetricsAaBeacon.beacon(beaconData.report);
    };
    dcBrowserMetrics.subscribe(aaBeaconSubscriber);

})(window['browser-metrics'], window['browser-metrics-aa-beacon']);
