(function (exports) {
    'use strict';

    const eventStream = [];
    const getEventStream = () => eventStream;

    function addReporter(reporter) {
        getEventStream().unshift({ addReporter: reporter });
    }

    function delegateTo(subscriber) {
        const eventStream = getEventStream();
        while (eventStream.length) {
            subscriber(eventStream.splice(0, 1)[0]);
        }
        eventStream.unshift = subscriber;
        eventStream.push = subscriber;
    }

    const getWindow = () => window;

    const getPerformance = () => getWindow().performance;

    function end({ key, entityId }) {
        const timestamp = getPerformance().now();
        console.debug("[BM]", key, "Probe.end", { timestamp });
        getEventStream().push({
            end: {
                key,
                timestamp,
                entityId,
            },
        });
    }

    const isNothing = obj => {
        return !obj || obj == null || obj === "null" || obj === "undefined";
    };

    const getDocument = () => getWindow().document;

    const satisfies = (selector, hasNone) => {
        var candidates = getDocument().querySelectorAll(selector);
        return (candidates.length &&
            (isNothing(hasNone) ||
                Array.prototype.every.call(candidates, function (element) {
                    return !element.querySelector(hasNone);
                })));
    };

    function Table() {
        this._ = {};
    }
    var _forEachRecursive = function (args) {
        var callback = args[0];
        var items = args[1];
        if (items instanceof Table) {
            if (args.length >= 3) {
                Object.keys(items._).forEach(function (key) {
                    _forEachRecursive([callback, items._[key], key].concat(args.slice(2)));
                });
            }
            else {
                Object.keys(items._).forEach(function (key) {
                    _forEachRecursive([callback, items._[key], key]);
                });
            }
        }
        else if (Array.isArray(items)) {
            callback.apply(null, [items].concat(args.slice(2)));
        }
    };
    Table.prototype.forEach = function (callback) {
        _forEachRecursive([callback, this]);
    };
    Table.prototype.add = function () {
        var obj = this;
        var cur = null;
        for (var i = 0; i < arguments.length; i++) {
            cur = arguments[i];
            if (i === arguments.length - 1 && Array.isArray(obj)) {
                obj.push(cur);
                break;
            }
            else if (i < arguments.length - 2 && !obj._.hasOwnProperty(cur)) {
                obj._[cur] = new Table();
            }
            else if (i === arguments.length - 2 && !obj._.hasOwnProperty(cur)) {
                obj._[cur] = [];
            }
            obj = obj._[cur];
        }
    };
    var _cleanUpRecursive = function (visited, keyToRemove) {
        if (visited.length === 0) {
            return;
        }
        var tuple = visited.pop();
        var key = tuple[0];
        var obj = tuple[1];
        if (key === keyToRemove) {
            _cleanUpRecursive(visited, key);
        }
        else if (obj._.hasOwnProperty(keyToRemove)) {
            delete obj._[keyToRemove];
        }
        if (Object.keys(obj).length === 0) {
            _cleanUpRecursive(visited, key);
        }
    };
    Table.prototype.remove = function () {
        var index;
        var mutated = false;
        var key = null;
        var obj = this;
        var visited = [[key, obj]];
        var cur = null;
        for (var i = 0; i < arguments.length; i++) {
            cur = arguments[i];
            if (Array.isArray(obj)) {
                index = obj.indexOf(cur);
                if (index > -1) {
                    obj.splice(index, 1);
                    if (obj.length === 0 && visited.length > 1) {
                        _cleanUpRecursive(visited, key);
                    }
                    mutated = true;
                }
            }
            else if (obj._.hasOwnProperty(cur)) {
                if (i === arguments.length - 1) {
                    delete obj._[cur];
                    if (Object.keys(obj).length === 0 && visited.length > 1) {
                        _cleanUpRecursive(visited, key);
                    }
                    mutated = true;
                }
                key = cur;
                obj = obj._[cur];
                visited.push([key, obj]);
            }
            else {
                break;
            }
        }
        return mutated;
    };
    Table.prototype.get = function (key) {
        if (this._.hasOwnProperty(key)) {
            return this._[key];
        }
        return [];
    };

    let mutationObserverInstance;
    const getMutationObserverInstance = () => {
        if (mutationObserverInstance) {
            return mutationObserverInstance;
        }
        var MutationObserver = getWindow().MutationObserver;
        var observer;
        var observing = false;
        var targets;
        targets = new Table();
        function addTarget(selector, hasNone, callback) {
            if (!observing) {
                observer.observe(getDocument(), {
                    attributes: true,
                    childList: true,
                    subtree: true,
                });
                observing = true;
            }
            targets.add(selector, hasNone, callback);
            return () => {
                targets.remove(selector, hasNone, callback);
            };
        }
        observer = new MutationObserver(function () {
            targets.forEach(function (callbacks, hasNone, selector) {
                if (satisfies(selector, hasNone)) {
                    callbacks.forEach(function (callback) {
                        callback();
                    });
                    targets.remove(selector, hasNone);
                }
            });
        });
        mutationObserverInstance = {
            addTarget,
        };
        return mutationObserverInstance;
    };

    function ElementHound(selectors, hasNones) {
        var dismiss;
        if (!selectors.forEach) {
            selectors = [selectors];
        }
        if (!isNothing(hasNones) && Array.isArray(hasNones)) {
            hasNones = hasNones.join(", ");
        }
        var result = new Promise(function (resolve, reject) {
            var requirementPromises = [];
            var requirementCleanups = [];
            selectors.forEach(function (selector) {
                var promise;
                if (!satisfies(selector, hasNones)) {
                    promise = new Promise(function (resolve) {
                        const cleanup = getMutationObserverInstance().addTarget(selector, hasNones, resolve);
                        requirementCleanups.push(cleanup);
                    });
                    requirementPromises.push(promise);
                }
            });
            var cleanup = function cleanup() {
                requirementCleanups.forEach(function (clean) {
                    clean();
                });
            };
            Promise.all(requirementPromises).then(cleanup).then(resolve, reject);
            dismiss = function dismiss() {
                cleanup();
                reject();
            };
        });
        result.dismiss = dismiss;
        return result;
    }

    var VISIBILITY_CHANGE = "visibilitychange";
    var ANIMATION_END = "animationend";
    var ANIMATION_CLASS_NAME = "browser-metrics-visibility-test";
    var ANIMATION_NAME = "browser-metrics-visibility-animation";
    var promise;
    var requireCssOnce = function () {
        var element = getDocument().createElement("style");
        var css = [
            "." + ANIMATION_CLASS_NAME + " {",
            "-webkit-animation-duration: 0.001s;",
            "animation-duration: 0.001s;",
            "-webkit-animation-name: " + ANIMATION_NAME + ";",
            "animation-name: " + ANIMATION_NAME + ";",
            "-webkit-animation-iteration-count: 1;",
            "animation-iteration-count: 1;",
            "}",
            "@keyframes " + ANIMATION_NAME + " {}",
            "@-webkit-keyframes " + ANIMATION_NAME + " {",
            "from {}",
            "to {}",
            "}",
        ].join("\n");
        element.type = "text/css";
        if (element.styleSheet) {
            element.styleSheet.cssText = css;
        }
        else {
            element.appendChild(getDocument().createTextNode(css));
        }
        getDocument().head.appendChild(element);
        requireCssOnce = function () { };
    };
    function queueAnimation() {
        requireCssOnce();
        getDocument().body.classList.add(ANIMATION_CLASS_NAME);
    }
    function requestImmediatePaint() {
        if (promise) {
            return promise;
        }
        var visibilityChanged = false;
        var onAnimationEnd;
        var onVisibilityChange;
        function cleanUp() {
            getDocument().body.classList.remove(ANIMATION_CLASS_NAME);
            getDocument().removeEventListener(VISIBILITY_CHANGE, onVisibilityChange);
            getDocument().removeEventListener(ANIMATION_END, onAnimationEnd);
            promise = null;
        }
        promise = new Promise(function (resolve, reject) {
            if (getDocument().visibilityState !== "visible") {
                reject();
            }
            else {
                onVisibilityChange = function () {
                    visibilityChanged = true;
                };
                onAnimationEnd = function (event) {
                    if (event.animationName !== ANIMATION_NAME) {
                        return;
                    }
                    if (visibilityChanged) {
                        reject();
                    }
                    else {
                        resolve();
                    }
                };
                getDocument().addEventListener(VISIBILITY_CHANGE, onVisibilityChange);
                getDocument().addEventListener(ANIMATION_END, onAnimationEnd);
                queueAnimation();
            }
        });
        promise.then(cleanUp, cleanUp);
        return promise;
    }

    function normalisedRules(rules) {
        if (!Array.isArray(rules)) {
            rules = [rules];
        }
        return rules.map(function (c) {
            return typeof c === "string" ? { selector: c, hasNone: null } : c;
        });
    }
    function isFuzzyRules(value) {
        return Array.isArray(value) || typeof value === "string";
    }
    function normalisedOptions(options) {
        if (isFuzzyRules(options)) {
            options = { rules: options };
        }
        options.rules = normalisedRules(options.rules);
        options.requirePaint =
            typeof options.requirePaint === "undefined"
                ? false
                : options.requirePaint;
        return options;
    }
    function find(options, callback) {
        options = normalisedOptions(options);
        var cancel = function () { };
        var result = new Promise(function (resolve, reject) {
            var executeOnCancel = [];
            var promises = options.rules.map(function (rule) {
                var hound = new ElementHound(rule.selector, rule.hasNone);
                executeOnCancel.push(function () {
                    hound.dismiss();
                });
                return hound;
            });
            cancel = function () {
                executeOnCancel.forEach(function (callback) {
                    callback();
                });
                reject();
            };
            Promise.all(promises)
                .then(function (results) {
            })
                .then(resolve, reject);
        });
        result.cancel = cancel;
        if (options.requirePaint) {
            result = result.then(requestImmediatePaint);
        }
        if (typeof callback === "function") {
            result.then(callback);
        }
        return result;
    }

    const document = getDocument();
    let done = false;
    switch (document.readyState) {
        case "interactive":
        case "complete":
            done = true;
            break;
        default:
            document.addEventListener("DOMContentLoaded", () => {
                getWindow().setTimeout(() => (done = true));
            }, {
                once: true,
            });
    }
    const isDOMContentLoadedDone = () => done;

    let currentFind;
    function reset() {
        currentFind = null;
    }
    function start(options) {
        var _a;
        const isInitial = "isInitial" in options
            ? options.isInitial
            : isDOMContentLoadedDone() === false;
        const threshold = "threshold" in options ? options.threshold : 1000;
        const reporters = "reporters" in options ? options.reporters : [];
        const timestamp = isInitial ? 0 : getPerformance().now();
        const timeout = (_a = options.timeout) !== null && _a !== void 0 ? _a : 60000;
        console.debug("[BM]", options.key, "Probe.start", { timestamp });
        getEventStream().push({
            start: {
                key: options.key,
                isInitial: isInitial,
                threshold: threshold,
                timestamp,
                reporters: Array.isArray(reporters) ? reporters : [reporters],
                entityId: options.entityId,
                waitForMarks: options.waitForMarks,
                timeout,
            },
        });
        if (currentFind) {
            currentFind.cancel();
            reset();
        }
        if (options.ready) {
            currentFind = find(options.ready);
            currentFind
                .then(function () {
                console.debug("[BM]", options.key, "Probe.end invoked by ready");
                end({ key: options.key });
            })
                .then(reset, reset);
        }
    }

    function subscribe(subscriber) {
        console.debug("[BM]", "Probe.subscribe", subscriber.name);
        getEventStream().push({ subscribe: subscriber });
    }

    if (window.define) {
        window.define("internal/browser-metrics", function () {
            return { start, end, addReporter, delegateTo, subscribe };
        });
    }
    console.debug("[BM]", "Probe loaded");

    exports.addReporter = addReporter;
    exports.delegateTo = delegateTo;
    exports.end = end;
    exports.start = start;
    exports.subscribe = subscribe;

    Object.defineProperty(exports, '__esModule', { value: true });

})(this["browser-metrics"] = this["browser-metrics"] || {});
