define('diagnostics/overview',
    [
        'aui',
        'jquery',
        'diagnostics/common',
        'diagnostics/field-filter',
        'diagnostics/interval-filter',
        'diagnostics/paging',
        'exports'
    ],
    function (AJS, $, Common, FieldFilter, IntervalFilter, paging, exports) {
        'use strict';

        /**
         * @param {jQuery|HTMLElement} $container
         */
        function AlertReportTable($container) {
            this.data = {};
            this.$container = $container;
            this.$table = $container.find('.alert-table-content');
            this.$paging = $("#diagnostics-paging");

            this.componentFilter = new FieldFilter($('#component-filter'), $('#component-filter-trigger'),
                this.changeFilter.bind(this) );
            this.intervalFilter = new IntervalFilter($('#interval-filter'), this.changeFilter.bind(this));
            this.nodeFilter = new FieldFilter($('#node-filter'), $('#node-filter-trigger'),
                this.changeFilter.bind(this) );
            this.pluginFilter = new FieldFilter($('#plugin-filter'), $('#plugin-filter-trigger'),
                this.changeFilter.bind(this) );
            this.severityFilter = new FieldFilter($('#severity-filter'), $('#severity-filter-trigger'),
                this.changeFilter.bind(this), false, ['Info', 'Warning', 'Error']);
            this.severityClass = {info: '', warning: 'aui-lozenge-moved', error: 'aui-lozenge-error'};

            JSON.parse($container.attr('data-components')).forEach(function(comp) {
                this.componentFilter.addValue(comp.id, comp.name);
            }.bind(this));
            JSON.parse($container.attr('data-nodes')).forEach(function(node) {
                this.nodeFilter.addValue(node);
            }.bind(this));
            JSON.parse($container.attr('data-plugins')).forEach(function(plugin) {
                this.pluginFilter.addValue(plugin.key, plugin.name);
            }.bind(this));

            this.renderFilters();
            paging.setChangeCallback(this.loadData.bind(this));
        }

        AlertReportTable.prototype.changeFilter = function() {
            // we have changed the filter, we go back to page 1
            paging.setPageData(0, paging.getPageData().limit);
            this.loadData();
        };

        AlertReportTable.prototype.loadData = function() {
            var self = this;
            var interval = this.intervalFilter.getInterval() || {
                to: Date.now(),
                from: 0
            };

            this.$container.find('#loadContainer').html(diagnostics.alertReport.loadingContent());
            const url = new URL(window.location.href);
            url.pathname = AJS.contextPath() + '/rest/diagnostics/latest/reports/alerts';
            url.search = new URLSearchParams({
                since: interval.from,
                until: interval.to,
                severity: Common.encodeSearchParamValueSet(self.severityFilter.getSelected()),
                node: Common.encodeSearchParamValueSet(self.nodeFilter.getSelected()),
                pluginKey: Common.encodeSearchParamValueSet(self.pluginFilter.getSelected()),
                componentId: Common.encodeSearchParamValueSet(self.componentFilter.getSelected()),
                start: paging.getPageData().start,
                limit: paging.getPageData().limit
            });
            fetch(url.toString()).then(response => response.json()).then((data) => self.setData(data));
        };

        AlertReportTable.prototype.render = function() {
            this.$container.find('.loading-banner-content').remove();
            if (!this.data || (this.data.size === 0 && !this.intervalFilter.getInterval())) {
                if (this.$container.find('.empty-banner-content').length === 0) {
                    this.$table.hide();
                    this.$container.append(diagnostics.alertReport.emptyBanner())
                }
                return;
            }

            var issuesById = {};
            if (this.data.issues) {
                issuesById = this.data.issues.reduce(function (accumulator, value) {
                    accumulator[value.id] = value;
                    return accumulator;
                }, {});
            }
            var rows = this.data.values.reduce(function (arr, alertCount) {
                var nodes = alertCount.nodeCounts.reduce(function (nodeArr, nodeCount) {
                    nodeArr.push(nodeCount.nodeName);
                    return nodeArr;
                }, []);
                alertCount.nodes = nodes;
                alertCount.count = alertCount.totalCount;
                arr.push(alertCount);
                return arr;
            }, []);

            this.$container.find('.empty-banner-content').remove();
            this.$table.show();
            var $tbody = $('<tbody/>');
            if (rows.length === 0) {
                $tbody.append(diagnostics.alertReport.noDataFoundRow())
            } else {
                var self = this;
                rows.forEach(function(row) {
                    $tbody.append(self.newTableRow(row, issuesById));
                });

                paging.renderHtmlContainer(this.$paging, this.data);
            }
            this.$container.find('tbody').replaceWith($tbody);
            $('a.nodes-tooltip').tooltip({gravity: 'n', html: true, className: "left-aligned-tipsy"});
        };

        AlertReportTable.prototype.newTableRow = function(rowData, issuesById) {
            var issue = issuesById[rowData.issueId];
            if (issue) {
                return diagnostics.alertReport.row({
                    alertCount: rowData.count,
                    component: issue.component.name,
                    contextPath: AJS.contextPath(),
                    issueId: issue.id,
                    issueSummary: issue.summary,
                    pluginKey: rowData.pluginKey,
                    pluginName: rowData.pluginName,
                    nodes: rowData.nodes,
                    severity: issue.severity,
                    severityClass: this.severityClass[issue.severity.toLowerCase()] || ''
                });
            }
        };

        AlertReportTable.prototype.renderFilters = function() {
            this.componentFilter.render();
            this.nodeFilter.render();
            this.pluginFilter.render();
            this.severityFilter.render();
        };

        AlertReportTable.prototype.setData = function(value) {
            this.data = value;
            this.render();
        };

        var reportTable = new AlertReportTable($('#diagnostics-container'));

        exports.reportTable = reportTable;
        exports.onReady = function() {
            reportTable.loadData();
            AJS.trigger('analyticsEvent', {name: "diagnostics.page.overview"});
        };
    }
);

AJS.toInit(function () {
  require('diagnostics/overview').onReady();
});
